/*
 * vector.cpp - vector class implementation
 *
 * Copyright (C) 2003, 2004, 2005, 2006, 2007 Stefan Jahn <stefan@lkcc.org>
 * Copyright (C) 2006, 2007 Gunther Kraut <gn.kraut@t-online.de>
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this package; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street - Fifth Floor,
 * Boston, MA 02110-1301, USA.  
 *
 * $Id: vector.cpp,v 1.40 2008/01/14 21:15:11 ela Exp $
 *
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <float.h>
#include <assert.h>

#include "complex.h"
#include "object.h"
#include "logging.h"
#include "strlist.h"
#include "vector.h"

// Constructor creates an unnamed instance of the vector class.
vector::vector () : object () {
  capacity = size = 0;
  data = NULL;
  dependencies = NULL;
  requested = 0;
}

/* Constructor creates an unnamed instance of the vector class with a
   given initial size. */
vector::vector (int s) : object () {
  assert (s >= 0);
  capacity = size = s;
  data = s > 0 ? (nr_complex_t *)
    calloc (capacity, sizeof (nr_complex_t)) : NULL;
  dependencies = NULL;
  requested = 0;
}

/* Constructor creates an unnamed instance of the vector class with a
   given initial size and content. */
vector::vector (int s, nr_complex_t val) : object () {
  assert (s >= 0);
  capacity = size = s;
  data = s > 0 ? (nr_complex_t *)
    calloc (capacity, sizeof (nr_complex_t)) : NULL;
  for (int i = 0; i < s; i++) data[i] = val;
  dependencies = NULL;
  requested = 0;
}

// Constructor creates an named instance of the vector class.
vector::vector (const char * n) : object (n) {
  capacity = size = 0;
  data = NULL;
  dependencies = NULL;
  requested = 0;
}

/* This constructor creates a named instance of the vector class with
   a given initial size. */
vector::vector (const char * n, int s) : object (n) {
  assert (s >= 0);
  capacity = size = s;
  data = s > 0 ? (nr_complex_t *)
    calloc (capacity, sizeof (nr_complex_t)) : NULL;
  dependencies = NULL;
  requested = 0;
}

/* The copy constructor creates a new instance based on the given
   vector object. */
vector::vector (const vector & v) : object (v) {
  size = v.size;
  capacity = v.capacity;
  data = (nr_complex_t *) malloc (sizeof (nr_complex_t) * capacity);
  memcpy (data, v.data, sizeof (nr_complex_t) * size);
  dependencies = v.dependencies ? new strlist (*v.dependencies) : NULL;
  requested = v.requested;
}

/* The assignment copy constructor creates a new instance based on the
   given vector object.  It copies the data only and leaves any other
   properties untouched. */
const vector& vector::operator=(const vector & v) {
  if (&v != this) {
    size = v.size;
    capacity = v.capacity;
    if (data) { free (data); data = NULL; }
    if (capacity > 0) {
      data = (nr_complex_t *) malloc (sizeof (nr_complex_t) * capacity);
      if (size > 0) memcpy (data, v.data, sizeof (nr_complex_t) * size);
    }
  }
  return *this;
}

// Destructor deletes a vector object.
vector::~vector () {
  if (data) free (data);
  if (dependencies) delete dependencies;
}

// output into file
void vector::saveData (FILE *f, bool isReal) {
  int n = sizeof(nr_complex_t);

  if(isReal) {
    n = sizeof(nr_double_t);
    nr_double_t * d = (double*)data;
    for (int i = 0; i < size; i++)
      d[i] = data[i].real();   // compress real-valued data
  }

  fwrite(data, n, size, f);
}

// load from file
bool vector::loadData (FILE *f, bool isReal) {
  int n = sizeof(nr_complex_t);
  if(isReal)
    n = sizeof(nr_double_t);

  if((int)fread(data, n, size, f) != size)   // read variable data
    return false;

  if(isReal) {
    nr_double_t * d = (double*)data;
    for (int i = size-1; i >= 0; i--)
      data[i] = d[i];   // decompress real-valued data
  }

  return true;
}

// Returns data dependencies.
strlist * vector::getDependencies (void) {
  return dependencies;
}

// Sets the data dependencies.
void vector::setDependencies (strlist * s) {
  if (dependencies) delete dependencies;
  dependencies = s;
}

/* The function appends a new complex data item to the end of the
   vector and ensures that the vector can hold the increasing number
   of data items. */
void vector::add (nr_complex_t c) {
  if (data == NULL) {
    size = 0; capacity = 64;
    data = (nr_complex_t *) malloc (sizeof (nr_complex_t) * capacity);
  }
  else if (size >= capacity) {
    capacity *= 2;
    data = (nr_complex_t *) realloc (data, sizeof (nr_complex_t) * capacity);
  }
  data[size++] = c;
}

/* This function appends the given vector to the vector. */
void vector::add (vector * v) {
  if (v != NULL) {
    if (data == NULL) {
      size = 0; capacity = v->getSize ();
      data = (nr_complex_t *) malloc (sizeof (nr_complex_t) * capacity);
    }
    else if (size + v->getSize () > capacity) {
      capacity += v->getSize ();
      data = (nr_complex_t *) realloc (data, sizeof (nr_complex_t) * capacity);
    }
    for (int i = 0; i < v->getSize (); i++) data[size++] = v->get (i);
  }
}

// Returns the complex data item at the given position.
nr_complex_t vector::get (int i) {
  return data[i];
}

void vector::set (nr_double_t d, int i) {
  data[i] = nr_complex_t (d);
}

void vector::set (const nr_complex_t z, int i) {
  data[i] = z;
}

// The function returns the current size of the vector.
int vector::getSize (void) {
  return size;
}

int vector::checkSizes (vector v1, vector v2) {
  if (v1.getSize () != v2.getSize ()) {
    logprint (LOG_ERROR, "vector '%s' and '%s' have different sizes\n",
	      v1.getName (), v2.getName ());
    return 0;
  }
  return 1;
}

nr_complex_t sum (vector v) {
  nr_complex_t result (0.0);
  for (int i = 0; i < v.getSize (); i++) result += v.get (i);
  return result;
}

vector abs (vector v) {
  vector result (v);
  for (int i = 0; i < v.getSize (); i++) result.set (abs (v.get (i)), i);
  return result;
}

vector norm (vector v) {
  vector result (v);
  for (int i = 0; i < v.getSize (); i++) result.set (norm (v.get (i)), i);
  return result;
}

vector arg (vector v) {
  vector result (v);
  for (int i = 0; i < v.getSize (); i++) result.set (arg (v.get (i)), i);
  return result;
}

vector real (vector v) {
  vector result (v);
  for (int i = 0; i < v.getSize (); i++) result.set (real (v.get (i)), i);
  return result;
}

vector imag (vector v) {
  vector result (v);
  for (int i = 0; i < v.getSize (); i++) result.set (imag (v.get (i)), i);
  return result;
}

vector sqrt (vector v) {
  vector result (v);
  for (int i = 0; i < v.getSize (); i++) result.set (sqrt (v.get (i)), i);
  return result;
}

vector vector::operator=(const nr_complex_t c) {
  for (int i = 0; i < size; i++) data[i] = c;
  return *this;
}

vector vector::operator=(const nr_double_t d) {
  for (int i = 0; i < size; i++) data[i] = d;
  return *this;
}

vector vector::operator+=(vector v) {
  int i, n, len = v.getSize ();
  assert (size % len == 0);
  for (i = n = 0; i < size; i++) { data[i] += v (n); if (++n >= len) n = 0; }
  return *this;
}

vector vector::operator+=(const nr_double_t d) {
  for (int i = 0; i < size; i++) data[i] += d;
  return *this;
}

vector operator+(vector v1, vector v2) {
  int len1 = v1.getSize (), len2 = v2.getSize ();
  vector result;
  if (len1 >= len2) {
    result  = v1;
    result += v2;
  } else {
    result  = v2;
    result += v1;
  }
  return result;
}

vector operator+(vector v, const nr_double_t d) {
  vector result (v);
  result += d;
  return result;
}

vector vector::operator-() {
  vector result (size);
  for (int i = 0; i < size; i++) result (i) = -data[i];
  return result;
}

vector vector::operator-=(vector v) {
  int i, n, len = v.getSize ();
  assert (size % len == 0);
  for (i = n = 0; i < size; i++) { data[i] -= v (n); if (++n >= len) n = 0; }
  return *this;
}

vector vector::operator-=(const nr_double_t d) {
  for (int i = 0; i < size; i++) data[i] -= d;
  return *this;
}

vector operator-(vector v1, vector v2) {
  int len1 = v1.getSize (), len2 = v2.getSize ();
  vector result;
  if (len1 >= len2) {
    result  = v1;
    result -= v2;
  } else {
    result  = -v2;
    result += v1;
  }
  return result;
}

vector operator-(vector v, const nr_double_t d) {
  vector result (v);
  result -= d;
  return result;
}

vector vector::operator*=(vector v) {
  int i, n, len = v.getSize ();
  assert (size % len == 0);
  for (i = n = 0; i < size; i++) { data[i] *= v (n); if (++n >= len) n = 0; }
  return *this;
}

vector vector::operator*=(const nr_double_t d) {
  for (int i = 0; i < size; i++) data[i] *= d;
  return *this;
}

vector operator*(vector v1, vector v2) {
  int len1 = v1.getSize (), len2 = v2.getSize ();
  vector result;
  if (len1 >= len2) {
    result  = v1;
    result *= v2;
  } else {
    result  = v2;
    result *= v1;
  }
  return result;
}

vector operator*(vector v, const nr_double_t d) {
  vector result (v);
  result *= d;
  return result;
}

vector operator*(const nr_double_t d, vector v) {
  return v * d;
}

vector vector::operator/=(vector v) {
  int i, n, len = v.getSize ();
  assert (size % len == 0);
  for (i = n = 0; i < size; i++) { data[i] /= v (n); if (++n >= len) n = 0; }
  return *this;
}

vector vector::operator/=(const nr_double_t d) {
  for (int i = 0; i < size; i++) data[i] /= d;
  return *this;
}

vector operator/(vector v1, vector v2) {
  int len1 = v1.getSize (), len2 = v2.getSize ();
  vector result;
  if (len1 >= len2) {
    assert (len1 % len2 == 0);
    result  = v1;
    result /= v2;
  } else {
    assert (len2 % len1 == 0);
    result  = 1 / v2;
    result *= v1;
  }
  return result;
}

vector operator/(vector v, const nr_double_t d) {
  vector result (v);
  result /= d;
  return result;
}

// This function reverses the order of the data list.
void vector::reverse (void) {
  nr_complex_t * buffer = (nr_complex_t *) malloc (sizeof (nr_complex_t) * size);
  for (int i = 0; i < size; i++) buffer[i] = data[size - 1 - i];
  free (data);
  data = buffer;
  capacity = size;
}

/* The function creates a linear stepped vector of values starting at
   the given start value, ending with the given stop value and
   containing points elements. */
vector linspace (nr_double_t start, nr_double_t stop, int points) {
  vector result (points);
  nr_double_t val, step = (stop - start) / (points - 1);
  for (int i = 0; i < points; i++) {
    val = start + (i * step);
    if (i != 0 && fabs (val) < fabs (step) / 4 && fabs (val) < NR_EPSI)
      val = 0.0;
    result.set (val, i);
  }
  return result;
}

/* The function creates a logarithmic stepped vector of values
   starting at the given start value, ending with the given stop value
   and containing points elements. */
vector logspace (nr_double_t start, nr_double_t stop, int points) {
  assert (start * stop > 0);
  vector result (points);
  nr_double_t step, first, last, d;

  // ensure the last value being larger than the first
  if (fabs (start) > fabs (stop)) {
    first = fabs (stop);
    last = fabs (start);
  }
  else {
    first = fabs (start);
    last = fabs (stop);
  }
  // check direction and sign of values
  d = fabs (start) > fabs (stop) ? -1 : 1;
  // compute logarithmic step size
  step = (log (last) - log (first)) / (points - 1);
  for (int i = 0, j = points - 1; i < points; i++, j--) {
    if (d > 0)
      result.set (start * exp (step * i), i);
    else
      result.set (stop * exp (step * i), j);
  }
  return result;
}
