/*
 * matrix.cpp - matrix class implementation
 *
 * Copyright (C) 2003, 2004, 2005, 2006, 2007 Stefan Jahn <stefan@lkcc.org>
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this package; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street - Fifth Floor,
 * Boston, MA 02110-1301, USA.  
 *
 * $Id: matrix.cpp,v 1.35 2008/01/13 10:50:16 ela Exp $
 *
 */

#include <config.h>

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "complex.h"
#include "vector.h"
#include "matrix.h"

/*!\brief Create an empty matrix

   Constructor creates an unnamed instance of the matrix class.
*/
matrix::matrix () {
  rows = 0;
  cols = 0;
  data = NULL;
}

/*!\brief Creates a square matrix

    Constructor creates an unnamed instance of the matrix class with a
    certain number of rows and columns.  Particularly creates a square matrix.  
    \param[in] s number of rows or colums of square matrix
    \todo Why not s const?
*/
matrix::matrix (int s)  {
  rows = cols = s;
  data = (s > 0) ? new nr_complex_t[s * s] : NULL;
}

/* \brief Creates a matrix

   Constructor creates an unnamed instance of the matrix class with a
   certain number of rows and columns.  
   \param[in] r number of rows
   \param[in] c number of column
   \todo Why not r and c constant
   \todo Assert r >= 0 and c >= 0
*/
matrix::matrix (int r, int c)  {
  rows = r;
  cols = c;
  data = (r > 0 && c > 0) ? new nr_complex_t[r * c] : NULL;
}

/* \brief copy constructor

   The copy constructor creates a new instance based on the given
   matrix object. 
   \todo Add assert tests
*/
matrix::matrix (const matrix & m) {
  rows = m.rows;
  cols = m.cols;
  data = NULL;

  // copy matrix elements
  if (rows > 0 && cols > 0) {
    data = new nr_complex_t[rows * cols];
    memcpy (data, m.data, sizeof (nr_complex_t) * rows * cols);
  }
}

/*!\brief Assignment operator
    
  The assignment copy constructor creates a new instance based on the
  given matrix object. 
  
  \param[in] m object to copy
  \return assigned object
  \note m = m is safe
*/
const matrix& matrix::operator=(const matrix & m) {
  if (&m != this) {
    rows = m.rows;
    cols = m.cols;
    if (data) { 
      delete[] data; 
      data = NULL; 
    }
    if (rows > 0 && cols > 0) {
      data = new nr_complex_t[rows * cols];
      memcpy (data, m.data, sizeof (nr_complex_t) * rows * cols);
    }
  }
  return *this;
}

/*!\bried Destructor
  
   Destructor deletes a matrix object.
*/
matrix::~matrix () {
  if (data) delete[] data;
}

/*!\brief  Returns the matrix element at the given row and column.
   \param[in] r row number
   \param[in] c column number
   \todo Why not inline and synonymous of ()
   \todo c and r const
*/
nr_complex_t matrix::get (int r, int c) {
  return data[r * cols + c];
}

/*!\brief Sets the matrix element at the given row and column.
   \param[in] r row number
   \param[in] c column number
   \param[in] z complex number to assign
   \todo Why not inline and synonymous of ()
   \todo r c and z const
*/
void matrix::set (int r, int c, nr_complex_t z) {
  data[r * cols + c] = z;
}

/*!\brief Matrix addition.
   \param[a] first matrix
   \param[b] second matrix
   \note assert same size
   \todo a and b are const
*/
matrix operator + (matrix a, matrix b) {
  assert (a.getRows () == b.getRows () && a.getCols () == b.getCols ());

  matrix res (a.getRows (), a.getCols ());
  for (int r = 0; r < a.getRows (); r++)
    for (int c = 0; c < a.getCols (); c++)
      res.set (r, c, a.get (r, c) + b.get (r, c));
  return res;
}

/*!\brief Matrix subtraction.
   \param[a] first matrix
   \param[b] second matrix
   \note assert same size
   \todo a and b are const
*/
matrix operator - (matrix a, matrix b) {
  assert (a.getRows () == b.getRows () && a.getCols () == b.getCols ());

  matrix res (a.getRows (), a.getCols ());
  for (int r = 0; r < a.getRows (); r++)
    for (int c = 0; c < a.getCols (); c++)
      res.set (r, c, a.get (r, c) - b.get (r, c));
  return res;
}

/*!\brief Unary minus. */
matrix matrix::operator - () {
  matrix res (getRows (), getCols ());
  int r, c, i;
  for (i = 0, r = 0; r < getRows (); r++)
    for (c = 0; c < getCols (); c++, i++)
      res.set (r, c, -data[i]);
  return res;
}

/*!\brief Matrix scaling complex version
   \param[in] a matrix to scale
   \param[in] z scaling complex
   \return Scaled matrix
   \todo Why not a and z const
*/
matrix operator * (matrix a, nr_complex_t z) {
  matrix res (a.getRows (), a.getCols ());
  for (int r = 0; r < a.getRows (); r++)
    for (int c = 0; c < a.getCols (); c++)
      res.set (r, c, a.get (r, c) * z);
  return res;
}

/*!\brief Matrix scaling complex version (different order)
   \param[in] a matrix to scale
   \param[in] z scaling complex
   \return Scaled matrix
   \todo Why not a and z const
   \todo Why not inline
*/
matrix operator * (nr_complex_t z, matrix a) {
  return a * z;
}

/*!\brief Matrix scaling complex version
   \param[in] a matrix to scale
   \param[in] d scaling real
   \return Scaled matrix
   \todo Why not d and a const
*/
matrix operator * (matrix a, nr_double_t d) {
  matrix res (a.getRows (), a.getCols ());
  for (int r = 0; r < a.getRows (); r++)
    for (int c = 0; c < a.getCols (); c++)
      res.set (r, c, a.get (r, c) * d);
  return res;
}

/*!\brief Matrix scaling real version (different order)
   \param[in] a matrix to scale
   \param[in] d scaling real
   \return Scaled matrix
   \todo Why not inline?
   \todo Why not d and a const
*/
matrix operator * (nr_double_t d, matrix a) {
  return a * d;
}

/*!\brief Matrix scaling division by complex version
   \param[in] a matrix to scale
   \param[in] z scaling complex
   \return Scaled matrix
   \todo Why not a and z const
*/
matrix operator / (matrix a, nr_complex_t z) {
  matrix res (a.getRows (), a.getCols ());
  for (int r = 0; r < a.getRows (); r++)
    for (int c = 0; c < a.getCols (); c++)
      res.set (r, c, a.get (r, c) / z);
  return res;
}

/*!\brief Matrix scaling division by real version
   \param[in] a matrix to scale
   \param[in] d scaling real
   \return Scaled matrix
   \todo Why not a and d const
*/
matrix operator / (matrix a, nr_double_t d) {
  matrix res (a.getRows (), a.getCols ());
  for (int r = 0; r < a.getRows (); r++)
    for (int c = 0; c < a.getCols (); c++)
      res.set (r, c, a.get (r, c) / d);
  return res;
}

/*! Matrix multiplication.
  
    Dumb and not optimized matrix multiplication
    \param[a] first matrix
    \param[b] second matrix
    \note assert compatibility
    \todo a and b are const
*/
matrix operator * (matrix a, matrix b) {
  assert (a.getCols () == b.getRows ());

  int r, c, i, n = a.getCols ();
  nr_complex_t z;
  matrix res (a.getRows (), b.getCols ());
  for (r = 0; r < a.getRows (); r++) {
    for (c = 0; c < b.getCols (); c++) {
      for (i = 0, z = 0; i < n; i++) z += a.get (r, i) * b.get (i, c);
      res.set (r, c, z);
    }
  }
  return res;
}

/*!\brief Complex scalar addition.
   \param[in] a matrix 
   \param[in] z complex to add
   \todo Move near other +
   \todo a and z are const
*/
matrix operator + (matrix a, nr_complex_t z) {
  matrix res (a.getRows (), a.getCols ());
  for (int r = 0; r < a.getRows (); r++)
    for (int c = 0; c < a.getCols (); c++)
      res.set (r, c, a.get (r, c) + z);
  return res;
}

/*!\brief Complex scalar addition different order.
   \param[in] a matrix 
   \param[in] z complex to add
   \todo Move near other +
   \todo a and z are const
   \todo Why not inline
*/
matrix operator + (nr_complex_t z, matrix a) {
  return a + z;
}

/*!\brief Real scalar addition.
   \param[in] a matrix 
   \param[in] d real to add
   \todo Move near other +
   \todo a and d are const
*/
matrix operator + (matrix a, nr_double_t d) {
  matrix res (a.getRows (), a.getCols ());
  for (int r = 0; r < a.getRows (); r++)
    for (int c = 0; c < a.getCols (); c++)
      res.set (r, c, a.get (r, c) + d);
  return res;
}

/*!\brief Real scalar addition different order.
   \param[in] a matrix 
   \param[in] d real to add
   \todo Move near other +
   \todo a and d are const
   \todo Why not inline
*/
matrix operator + (nr_double_t d, matrix a) {
  return a + d;
}

/*!\brief Complex scalar substraction
   \param[in] a matrix 
   \param[in] z complex to add
   \todo Move near other +
   \todo a and z are const
   \todo Why not inline
*/
matrix operator - (matrix a, nr_complex_t z) {
  return -z + a;
}

/*!\brief Complex scalar substraction different order
   \param[in] a matrix 
   \param[in] z complex to add
   \todo Move near other +
   \todo a and z are const
   \todo Why not inline
*/
matrix operator - (nr_complex_t z, matrix a) {
  return -a + z;
}

/*!\brief Real scalar substraction
   \param[in] a matrix 
   \param[in] z real to add
   \todo Move near other +
   \todo a and z are const
   \todo Why not inline
*/
matrix operator - (matrix a, nr_double_t d) {
  return -d + a;
}

/*!\brief Real scalar substraction different order
   \param[in] a matrix 
   \param[in] z real to add
   \todo Move near other +
   \todo a and z are const
   \todo Why not inline
*/
matrix operator - (nr_double_t d, matrix a) {
  return -a + d;
}

/*!\brief Create identity matrix with specified number of rows and columns.
   \param[in] rs row number
   \param[in] cs column number
   \todo Avoid res.get* 
   \todo Use memset
   \todo rs, cs are const
*/
matrix eye (int rs, int cs) {
  matrix res (rs, cs);
  for (int r = 0; r < res.getRows (); r++)
    for (int c = 0; c < res.getCols (); c++)
      if (r == c) res.set (r, c, 1);
  return res;
}

/*!\brief Create a square identity matrix
   \param[in] s row or column number of square matrix
   \todo Do not by lazy and implement it
   \todo s is const
*/
matrix eye (int s) {
  return eye (s, s);
}

/*!\brief Create a diagonal matrix from a vector
   \param[in] diag vector to write on the diagonal
   \todo diag is const
*/
matrix diagonal (vector diag) {
  int size = diag.getSize ();
  matrix res (size);
  for (int i = 0; i < size; i++) res (i, i) = diag (i);
  return res;
}

/*!\brief Compute inverse matrix using Gauss-Jordan elimination
   
   Compute inverse matrix of the given matrix by Gauss-Jordan
   elimination. 
   \todo a const?
   \todo static?
   \note assert non singulat matix
   \param[in] a matrix to invert
*/
matrix inverse (matrix a) {
  nr_double_t MaxPivot;
  nr_complex_t f;
  matrix b, e;
  int i, c, r, pivot, n = a.getCols ();

  // create temporary matrix and the result matrix
  b = matrix (a);
  e = eye (n);

  // create the eye matrix in 'b' and the result in 'e'
  for (i = 0; i < n; i++) {
    // find maximum column value for pivoting
    for (MaxPivot = 0, pivot = r = i; r < n; r++) {
      if (abs (b (r, i)) > MaxPivot) {
	MaxPivot = abs (b (r, i));
	pivot = r;
      }
    }
    // exchange rows if necessary
    assert (MaxPivot != 0); // singular matrix
    if (i != pivot) {
      b.exchangeRows (i, pivot);
      e.exchangeRows (i, pivot);
    }

    // compute current row
    for (f = b (i, i), c = 0; c < n; c++) {
      b (i, c) /= f;
      e (i, c) /= f;
    }

    // compute new rows and columns
    for (r = 0; r < n; r++) {
      if (r != i) {
	for (f = b (r, i), c = 0; c < n; c++) {
	  b (r, c) -= f * b (i, c);
	  e (r, c) -= f * e (i, c);
	}
      }
    }
  }
  return e;
}

/*!\brief S params to S params

  Convert scattering parameters with the reference impedance 'zref'
  to scattering parameters with the reference impedance 'z0'. 

  Detail are given in [1], under equation (32)
  
  New scatering matrix \f$S'\f$ is:
  \f[
  S'=A^{-1}(S-\Gamma^+)(I-\Gamma S)^{-1}A^+
  \f]
  Where x^+ is the adjoint (or complex tranposate) of x,
  I the identity matrix and \f$A\f$ is diagonal the matrix such as:
  \f$   \Gamma_i= r_i \f$ and \f$A\f$ the diagonal matrix such
  as:
  \f[
  A_i =  \frac{(1-r_i^*)\sqrt{|1-r_ir_i^*|}}{|1-r_i|}
  \f]
  Where \f$x*\f$ is the complex conjugate of \f$x\f$ 
  and \f$r_i\f$ is wave reflexion coefficient of \f$Z_i'\f$ with respect 
  to \f$Z_i^*\f$ (where \f$Z_i'\f$ is the new impedance and 
  \f$Z_i\f$ is the old impedance), ie:
  \f[
  r_i = \frac{Z_i'-Z_i}{Z_i'-Z_i^*}
  \f]
  
  \param[in] s original S matrix
  \param[in] zref original reference impedance
  \param[in] z0 new reference impedance
  \bug This formula is valid only for real z!
  \todo Correct documentation about standing waves [1-4]
  \todo Implement Speciale implementation [2-3] if applicable
  \return Renormalized scattering matrix
  \todo s, zref and z0 const
*/
matrix stos (matrix s, vector zref, vector z0) {
  int d = s.getRows ();
  matrix e, r, a;

  assert (d == s.getCols () && d == z0.getSize () && d == zref.getSize ());

  e = eye (d);
  r = diagonal ((z0 - zref) / (z0 + zref));
  a = diagonal (sqrt (z0 / zref) / (z0 + zref));
  return inverse (a) * (s - r) * inverse (e - r * s) * a;
}

/*!\brief S renormalization with all part identic and real
  \param[in] s original S matrix
  \param[in] zref original reference impedance
  \param[in] z0 new reference impedance
  \todo Why not inline
  \return Renormalized scattering matrix
  \todo s, zref and z0 const
*/ 
matrix stos (matrix s, nr_double_t zref, nr_double_t z0) {
  int d = s.getRows ();
  return stos (s, vector (d, zref), vector (d, z0));
}

/*!\brief The function swaps the given rows with each other.
  \param[in] r1 source row
  \param[in] r2 destination row
  \note assert not out of bound r1 and r2
  \todo r1 and r2 const
*/
void matrix::exchangeRows (int r1, int r2) {
  nr_complex_t * s = new nr_complex_t[cols];
  int len = sizeof (nr_complex_t) * cols;
  
  assert (r1 >= 0 && r2 >= 0 && r1 < rows && r2 < rows);

  memcpy (s, &data[r1 * cols], len);
  memcpy (&data[r1 * cols], &data[r2 * cols], len);
  memcpy (&data[r2 * cols], s, len);
  delete[] s;
}

/*!\brief The function swaps the given column with each other.
  \param[in] c1 source column
  \param[in] c2 destination column
  \note assert not out of bound r1 and r2
  \todo c1 and c2 const
*/
void matrix::exchangeCols (int c1, int c2) {
  nr_complex_t s;

  assert (c1 >= 0 && c2 >= 0 && c1 < cols && c2 < cols);

  for (int r = 0; r < rows * cols; r += cols) {
    s = data[r + c1];
    data[r + c1] = data[r + c2];
    data[r + c2] = s;
  }
}
