/*
 * matrix.h - matrix class definitions
 *
 * Copyright (C) 2003, 2004, 2005, 2006, 2007 Stefan Jahn <stefan@lkcc.org>
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this package; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street - Fifth Floor,
 * Boston, MA 02110-1301, USA.  
 *
 * $Id: matrix.h,v 1.24 2008/01/13 10:50:20 ela Exp $
 *
 */

#ifndef __MATRIX_H__
#define __MATRIX_H__

class vector;
class matrix;


/*!\brief Dense complex matrix class */
class matrix
{
 public:
  matrix ();
  matrix (int);
  matrix (int, int);
  matrix (const matrix &);
  const matrix& operator = (const matrix &);
  ~matrix ();
  nr_complex_t get (int, int);
  void set (int, int, nr_complex_t);
  int getCols (void) { return cols; }
  int getRows (void) { return rows; }
  nr_complex_t * getData (void) { return data; }
  void exchangeRows (int, int);
  void exchangeCols (int, int);

  // operator functions
  friend matrix operator + (matrix, matrix);
  friend matrix operator + (nr_complex_t, matrix);
  friend matrix operator + (matrix, nr_complex_t);
  friend matrix operator + (nr_double_t, matrix);
  friend matrix operator + (matrix, nr_double_t);
  friend matrix operator - (matrix, matrix);
  friend matrix operator - (nr_complex_t, matrix);
  friend matrix operator - (matrix, nr_complex_t);
  friend matrix operator - (nr_double_t, matrix);
  friend matrix operator - (matrix, nr_double_t);
  friend matrix operator / (matrix, nr_complex_t);
  friend matrix operator / (matrix, nr_double_t);
  friend matrix operator * (nr_complex_t, matrix);
  friend matrix operator * (matrix, nr_complex_t);
  friend matrix operator * (nr_double_t, matrix);
  friend matrix operator * (matrix, nr_double_t);
  friend matrix operator * (matrix, matrix);

  // intrinsic operator functions
  matrix operator  - ();

  // other operations
  friend matrix eye (int, int);
  friend matrix diagonal (vector);
  friend matrix inverse (matrix);
  friend matrix stos (matrix, nr_double_t, nr_double_t z0 = 50.0);
  friend matrix stos (matrix, vector, vector);

  /*! \brief Read access operator 
      \param[in] r: row number (from 0 like usually in C)
      \param[in] c: column number (from 0 like usually in C)
      \return Cell in the row r and column c
      \todo: Why not inline
      \todo: Why not r and c not const
      \todo: Create a debug version checking out of bound (using directly assert)
  */  
  nr_complex_t  operator () (int r, int c) const { return data[r * cols + c]; }

 private:
  /*! Number of colunms */
  int cols;
  /*! Number of rows */
  int rows;
  /*! Matrix data */
  nr_complex_t * data;
};

#endif /* __MATRIX_H__ */
