/*
 * qucs_producer.cpp - the Qucs netlist producer
 *
 * Copyright (C) 2004, 2005, 2006, 2007 Stefan Jahn <stefan@lkcc.org>
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this package; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street - Fifth Floor,
 * Boston, MA 02110-1301, USA.  
 *
 * $Id: qucs_producer.cpp,v 1.24 2007/09/16 16:49:41 ela Exp $
 *
 */

#include <config.h>

#include <io.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#include "object.h"
#include "complex.h"
#include "dataset.h"
#include "netdefs.h"
#include "check_spice.h"
#include "check_vcd.h"
#include "hash.h"
#include "touchstone_producer.h"

/* Global variables. */
FILE * qucs_out = NULL;
int    qucs_append = 0;
const char * qucs_gnd = "gnd";

qucs::hash<struct node_t> qucs_nodes;

/* Looks through the given list if there is such a node already in the
   list and returns non-zero if so, otherwise the function returns
   zero. */
int qucs_find_node (struct node_t * root, char * node) {
  for (struct node_t * n = root; n; n = n->next) {
    if (!strcmp (node, n->node)) return 1;
  }
  return 0;
}

/* Adds the given node list to the overall node list without
   duplicating the nodes. */
void qucs_add_nodes (struct node_t * node) {
  while (node) {
    if (!qucs_nodes.get (node->node))
      qucs_nodes.put (node->node, node);
    node = node->next;
  }
}

/* Deletes the given node list. */
void qucs_free_nodes (struct node_t * node) {
  struct node_t * n;
  for ( ; node; node = n) {
    n = node->next;
    if (node->node) free (node->node);
    free (node);
  }
}

/* Prints value representation. */
static void netlist_list_value (struct value_t * value, FILE *f) {
  if (value == NULL)
    fprintf (f, "[]");
  else if (value->ident)
    fprintf (f, "%s", value->ident);
  else if (value->next) {
    fprintf (f, "[");
    for (; value != NULL; value = value->next)
      fprintf (f, "%g%s", value->value, value->next ? ";" : "");
    fprintf (f, "]");
  }
  else {
    if (value->scale) {
      if ((value->scale[0] == 'm') && (value->unit == 0))
        fprintf (f, "%g", value->value*1e-3);  // a single "m" is meter, so use exponent
      else
        fprintf (f, "%g%s", value->value, value->scale);
    }
    else
      fprintf (f, "%g", value->value);
    if (value->unit)
      fprintf (f, "%s", value->unit);
  }
}

/* Goes recursively through the netlist and applies additional
   reference node modifications to subcircuits and their instances. */
static void netlist_fix_reference (struct definition_t * root) {
  struct definition_t * def;
  struct node_t * n;

  // go through definitions
  for (def = root; def != NULL; def = def->next) {
    if (!strcmp (def->type, "Sub")) { // subcircuit instances
      n = create_node ();
      n->node = _strdup (qucs_gnd);
      n->next = def->nodes;
      def->nodes = n;
    }
    else if (def->sub != NULL) {     // subcircuit definitions
      n = create_node ();
      n->node = _strdup (qucs_gnd);
      n->next = def->nodes;
      def->nodes = n;
      netlist_fix_reference (def->sub);
    }
  }
}

/* This function tries to find the given property in the key/value
   pairs of the given netlist entry and returns NULL if there is no
   such property. */
static struct pair_t *
qucslib_find_prop (struct definition_t * def, const char * key) {
  struct pair_t * pair;
  for (pair = def->pairs; pair; pair = pair->next) {
    if (!strcmp (pair->key, key))
      return pair;
  }
  return NULL;
}

// ---------------------------------------------------------------
// This function is the overall Qucs library producer.
bool qucslib_producer (char *outfile)
{
  int fileExist;
  char *p, *name;
  const char *qmodel;
  struct node_t *node;
  struct pair_t *pair;
  struct definition_t *def, *rdef;

  fileExist = _access(outfile, 0);

  if(qucs_append)
    qmodel = "ab";
  else
    qmodel = "wb"; // "binary" avoids Windows linefeed

  qucs_out = open_file (outfile, qmodel);
  if(qucs_out == NULL)
    return false;

  // library name is base name of output file
  #ifdef WIN32
  name = strrchr(outfile, '\\');
  #else
  name = strrchr(outfile, '/');
  #endif
  if(name)  // cut off path
    name++;
  else
    name = outfile;
  name = _strdup(name);
  p = strrchr(name, '.');
  if(p)  // cut off suffix
    *p = 0;

  if((qucs_append == 0) || (fileExist == -1))
    fprintf (qucs_out, QUCSLIBRARY_ID PACKAGE_VERSION " \"%s\">\n", name);

  // output all model cards
  for(def = device_root; def; def = def->next) {
    qmodel = def->type;
    if(strcmp(qmodel, "BJT") == 0) {
      pair = qucslib_find_prop (def, "Cjs");
      if(spice_evaluate_value (pair->value) == 0.0)
        qmodel = "_BJT";
    }

    fprintf (qucs_out, "\n<Component %s>\n", def->instance);
    fprintf (qucs_out, "  <Description>\n");
    if (def->description)
      fprintf (qucs_out, "    %s\n", def->description);
    fprintf (qucs_out, "  </Description>\n");
    fprintf (qucs_out, "  <Model>\n");
    fprintf (qucs_out, "    %s %s_ 1 0 0 10 12 0 0", qmodel, def->instance);
    for (pair = def->pairs; pair!=0; pair=pair->next) {
      fprintf (qucs_out, " \"");
      netlist_list_value (pair->value, qucs_out);
      fprintf (qucs_out, "\" 0");
    }
    fprintf (qucs_out, "\n");
    fprintf (qucs_out, "  </Model>\n");
    fprintf (qucs_out, "</Component>\n");
  }

  p = name;
  while(*(p++))
    if(p[-1] == ' ') // there has to be no spaces in ".Def:"
      p[-1] = '_';

  // output all subcircuits
  for(rdef = definition_root; rdef != NULL; rdef = rdef->next) {
    def = rdef->sub;
    if(def == 0)  // not a subcircuit?
      continue;

    fprintf (qucs_out, "\n<Component %s>\n", rdef->instance);
    fprintf (qucs_out, "  <Description>\n");
    if (rdef->description)
      fprintf (qucs_out, "    %s\n", rdef->description);
    fprintf (qucs_out, "  </Description>\n");
    fprintf (qucs_out, "  <Model>\n");
    fprintf (qucs_out, "    .Def:L%s_%s", name, rdef->instance);
    for (node = rdef->nodes; node != NULL; node = node->next)
      fprintf (qucs_out, " %s", node->node);
    fprintf (qucs_out, "\n");

    for(; def != NULL; def = def->next) {
      if(def->sub || (strcmp(def->type, "Sub") == 0)) {
        fprintf(stderr, "ERROR: Nested subcircuits are not allowed in library.");
        free(name);
        return false;
      }
      if(def->define == NULL)
        continue;

      fprintf (qucs_out, "      ");
      if(def->action && (strcmp(def->type, "EDD") != 0))
        fprintf (qucs_out, ".");
      fprintf (qucs_out, "%s:%s%s",
            def->type, isdigit (def->instance[0]) ? "X" : "", def->instance);

      for(node = def->nodes; node != NULL; node = node->next)
        fprintf (qucs_out, " %s", node->node);

      for(pair = def->pairs; pair != NULL; pair = pair->next) {
        fprintf (qucs_out, " \"");
        netlist_list_value (pair->value, qucs_out);
        fprintf (qucs_out, "\"");
      }
      fprintf (qucs_out, "\n");
    }

    fprintf (qucs_out, "    .Def:End\n");
    fprintf (qucs_out, "  </Model>\n");
    fprintf (qucs_out, "</Component>\n");
  }

  fclose (qucs_out);
  free(name);
  return true;
}

// ---------------------------------------------------------------
#define NEXT_POS(x,y)   {x += 100; \
                         if(x > 700) { \
                           x  = 50; \
                           y += 140; \
                         }}

// Creates schematic recursively because of embedded subcircuits.
bool createSubcircuitSchematic(char *outfile, struct definition_t *subRoot, bool isSubcircuit)
{
  const char *s;
  int i, num, x, y, dx, dy;
  struct node_t *node;
  struct pair_t *pair;
  struct definition_t *def;

  if(subRoot != definition_root) {  // don't check for main schematic
    struct stat st;
    i = stat(outfile, &st);
    if(i == 0) {
      fprintf(stderr, "ERROR: Subcircuit schematic '%s' already exists.", outfile);
      return false;
    }
  }

  FILE *f = open_file (outfile, "wb"); // "binary" avoids Windows linefeed
  if(f == NULL)
    return false;

  fprintf (f, QUCSFILE_ID PACKAGE_VERSION ">\n");
  fprintf (f, "<Properties>\n</Properties>\n");
  fprintf (f, "<Symbol>\n</Symbol>\n");
  fprintf (f, "<Components>\n");

  x = y = 50;
  def = subRoot;

  if(isSubcircuit) {
    def = def->sub;
    i = 1;  // exactly one subcircuit, so output subcircuit ports
    for (node = subRoot->nodes; node != NULL; node = node->next) {
      fprintf (f,
        "Port P%s 1 %d %d -23 12 0 0 \"%d\" 0 \"analog\" 0 \"TESTPOINT\" 0\n",
        node->node, x, y, i);
      i++;
      NEXT_POS(x,y)
    }
  }

  for (; def != NULL; def = def->next) {
    if(def->sub) {    // Is it a subcircuit?
      char fileName[128];
      sprintf(fileName, "%s.sch", def->instance);
      if(!createSubcircuitSchematic(fileName, def, true))
        return false;
      continue;
    }
    if(def->define == NULL)
      continue;


    if((strcmp(def->type, "EDD") == 0) || (strcmp(def->type, "Sub") == 0)) {
      // Create EDD or subcircuit instance.
      num = 1;
      for (node = def->nodes; node != NULL; node = node->next)
        num++;
      num /= 2;
      x += num*60/2 - 50;
      fprintf (f, "%s %s%s 1 %d %d 10 15 0 1",
          def->type, isdigit (def->instance[0]) ? "X" : "", def->instance, x, y);
      x += num*60/2 - 90;
    }
    else
      fprintf (f, "%s%s %s%s 1 %d %d 10 15 0 0", def->action ? "." : "",
          def->type, isdigit (def->instance[0]) ? "X" : "", def->instance, x, y);
    for (pair = def->pairs; pair != NULL; pair = pair->next) {
      fprintf (f, " \"");
      netlist_list_value (pair->value, f);
      fprintf (f, "\" 0");
    }
    fprintf (f, "\n");

    NEXT_POS(x,y)
  }

  fprintf (f, "</Components>\n");
  fprintf (f, "<Wires>\n");

  x = y = 50;

  def = subRoot;
  if(isSubcircuit) {
    def = def->sub;
    // labels for subcircuit ports
    for (node = subRoot->nodes; node != NULL; node = node->next) {
      s = node->node;
      if(*s == '_')  s++;
      fprintf (f, "%d %d %d %d \"%s\" %d %d 0 \"\"\n",
               x, y, x, y, s, x+10, y-30);
      NEXT_POS(x,y)
    }
  }

  // output labels to connect components
  for (; def != NULL; def = def->next) {
    if(def->sub)    // Is it a subcircuit?
      continue;
    if (def->define == NULL)
      continue;

    if((strcmp(def->type, "EDD") == 0) || (strcmp(def->type, "Sub") == 0)) {
      // Create EDD or subcircuit instance.
      num = 1;
      for (node = def->nodes; node != NULL; node = node->next)
        num++;
      num /= 2;
      x += num*60/2 - 50;

      i = 0;
      for (node = def->nodes; node != NULL; node = node->next) {
        dx = (i&-2) + 1-num;
        if(num > 4)
          dx *= 20;
        else
          dx *= 30;
        dy = 30;
        if(i & 1)
          dy = -30;
        s = node->node;
        if(*s == '_')
          s++;
        fprintf (f, "%d %d %d %d \"%s\" %d %d 0 \"\"\n",
                 x+dx, y+dy, x+dx, y+dy, s, x+dx+10, y+dy-30);
        i++;
      }
      x += num*60/2 - 90;
    }
    else {  // standard component
      i = 0;
      for (node = def->nodes; node != NULL; node = node->next) {
        dx = def->define->portx[i];
        dy = def->define->porty[i];
        s = node->node;
        if(*s == '_')
          s++;
        fprintf (f, "%d %d %d %d \"%s\" %d %d 0 \"\"\n",
                 x+dx, y+dy, x+dx, y+dy, s, x+dx+10, y+dy-30);
        i++;
      }
    }

    NEXT_POS(x,y)
  }

  fprintf (f, "</Wires>\n");
  fprintf (f, "<Diagrams>\n</Diagrams>\n");
  fprintf (f, "<Paintings>\n</Paintings>\n");
  fclose (f);
  return true;
}

// ---------------------------------------------------------------
// This function is the overall schematic producer.
bool qucsschematic_producer(char *outfile)
{
  const char *s;
  struct pair_t *pair;
  struct definition_t *def;
  int x, y;

  // test if there's something different than a model definition
  for(def = definition_root; def != NULL; def = def->next)
    if(strcmp(def->type, "MODEL")) {
      // create subcircuit
      bool isSub = ((definition_root->next == 0) && definition_root->sub);
      if(!createSubcircuitSchematic(outfile, definition_root, isSub))
        return false;
      return true;
    }

  qucs_out = open_file (outfile, "wb"); // "binary" avoids Windows linefeed
  if(qucs_out == NULL)
    return false;

  fprintf (qucs_out, QUCSFILE_ID PACKAGE_VERSION ">\n");
  fprintf (qucs_out, "<Properties>\n</Properties>\n");
  fprintf (qucs_out, "<Symbol>\n</Symbol>\n");
  fprintf (qucs_out, "<Components>\n");

  // output one component for each model definition
  x = y = 50;
  for (def = device_root; def; def = def->next) {

    s = def->type;
    if(strcmp(s, "BJT") == 0) {
      pair = qucslib_find_prop (def, "Cjs");
      if(spice_evaluate_value (pair->value) == 0.0)  // substrate not used?
        s = "_BJT";  // BJT without substrate
    }

    fprintf (qucs_out, "%s %s_ 1 %d %d 10 12 0 0", def->type, def->instance, x, y);
    for (pair = def->pairs; pair!=0; pair=pair->next) {
      fprintf (qucs_out, " \"");
      netlist_list_value (pair->value, qucs_out);
      fprintf (qucs_out, "\" 0");
    }
    fprintf (qucs_out, "\n");

    NEXT_POS(x,y)
  }

  fprintf (qucs_out, "</Components>\n");
  fprintf (qucs_out, "<Wires>\n</Wires>\n");
  fprintf (qucs_out, "<Diagrams>\n</Diagrams>\n");
  fprintf (qucs_out, "<Paintings>\n</Paintings>\n");
  fclose (qucs_out);
  return true;
}

// ---------------------------------------------------------------
// This function is the Qucs dataset producer for VCD files.
void qucsdata_producer_vcd (void) {
  // write prologue
  fputs (DATA_FORMAT_ID, qucs_out);
  int n = QUCS_HEX_VERSION;
  fwrite(&n, sizeof(int), 1, qucs_out);

  int count = 0;   // count number of bytes for header
  fwrite(&count, sizeof(int), 1, qucs_out);   // size of header (placeholder)

  int headerSize = 0;
  struct dataset_variable * ds;
  struct dataset_value * dv;

  // write header
  for (ds = dataset_root; ds; ds = ds->next) {
    if (!ds->output || ds->type == DATA_UNKNOWN)
      continue;
    if (ds->ident[0] == '_')
      continue;
    if(ds->type == DATA_INDEPENDENT) {
      n = VARTYPE_IS_VAR + VARTYPE_IS_REAL;
      fwrite(&n, sizeof(int), 1, qucs_out);
      n = ds->size;
      fwrite(&n, sizeof(int), 1, qucs_out);
      fputs (ds->ident, qucs_out);
      fputc (0, qucs_out);
      headerSize += strlen(ds->ident) + 9;  // +9 -> type (4Bytes), size (4Bytes), null-termination (1Byte)
    }
    else if (ds->type == DATA_DEPENDENT) {
      n = VARTYPE_IS_VAR + VARTYPE_IS_DEP;
      if(!ds->isreal)
        n += VARTYPE_IS_DIGI;
      fwrite(&n, sizeof(int), 1, qucs_out);
      if(ds->isreal)
        n = ds->size;
      else {
        n = 0;
        for(dv = ds->values; dv; dv = dv->next)
          n += strlen(dv->value) + 1;
      }
      fwrite(&n, sizeof(int), 1, qucs_out);
      fputs (ds->ident, qucs_out);
      if(ds->isreal)
        fputs (".R", qucs_out);
      else
        fputs (".X", qucs_out);
      fputc (0, qucs_out);
      fputs (ds->dependencies, qucs_out);
      fputc (0, qucs_out);
      headerSize += strlen(ds->ident) + strlen(ds->dependencies) + 12;
    }
  }
  
  // print variables
  double d;
  for (ds = dataset_root; ds; ds = ds->next) {
    if (!ds->output || ds->type == DATA_UNKNOWN)
      continue;
    if (ds->ident[0] == '_')
      continue;
    if(ds->isreal || (ds->type == DATA_INDEPENDENT))
      for (dv = ds->values; dv; dv = dv->next) {
        d = strtod(dv->value, 0);
        fwrite(&d, sizeof(double), 1, qucs_out);
      }
	else
      for (dv = ds->values; dv; dv = dv->next) {
        fputs (dv->value, qucs_out);
        fputc (0, qucs_out);
      }
  }

  fseek(qucs_out, 12, SEEK_SET);
  fwrite(&headerSize, sizeof(int), 1, qucs_out);   // size of header
}
