/*
 * vector.h - vector class definitions
 *
 * Copyright (C) 2003, 2004, 2005, 2006, 2007 Stefan Jahn <stefan@lkcc.org>
 * Copyright (C) 2006 Gunther Kraut <gn.kraut@t-online.de>
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this package; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street - Fifth Floor,
 * Boston, MA 02110-1301, USA.  
 *
 * $Id: vector.h,v 1.33 2008/01/14 21:15:11 ela Exp $
 *
 */

#ifndef __VECTOR_H__
#define __VECTOR_H__

#include <math.h>

#include "precision.h"
#include "object.h"

class strlist;
class vector;

vector linspace (nr_double_t, nr_double_t, int);
vector logspace (nr_double_t, nr_double_t, int);

class vector : public object
{
 public:
  vector ();
  vector (const char *);
  vector (int);
  vector (int, nr_complex_t);
  vector (const char *, int);
  vector (const vector &);
  const vector& operator = (const vector &);
  ~vector ();
  void add (nr_complex_t);
  void add (vector *);
  nr_complex_t get (int);
  void set (nr_double_t, int);
  void set (const nr_complex_t, int);
  int getSize (void);
  int checkSizes (vector, vector);
  int getRequested (void) { return requested; }
  void setRequested (int n) { requested = n; }
  void reverse (void);
  strlist * getDependencies (void);
  void setDependencies (strlist *);

  void saveData (FILE *, bool isReal=false);
  bool loadData (FILE *, bool isReal=false);

  friend nr_complex_t sum (vector);

  // vector manipulations
  friend vector real   (vector);  // the real part
  friend vector imag   (vector);  // the imaginary part
  friend vector norm   (vector);  // the square of the magnitude
  friend vector arg    (vector);  // the angle in the plane

  // overloaded math functions
  friend vector abs     (vector);
  friend vector sqrt    (vector);

  // operator functions
  friend vector operator + (vector, vector);
  friend vector operator + (vector, const nr_double_t);
  friend vector operator - (vector, vector);
  friend vector operator - (vector, const nr_double_t);
  friend vector operator - (const nr_double_t, vector);
  friend vector operator * (vector, vector);
  friend vector operator * (vector, const nr_double_t);
  friend vector operator / (vector, vector);
  friend vector operator / (vector, const nr_double_t);

  // assignment operations
  vector operator  - ();
  vector operator  = (const nr_complex_t);
  vector operator  = (const nr_double_t);
  vector operator += (vector);
  vector operator += (const nr_double_t);
  vector operator -= (vector);
  vector operator -= (const nr_double_t);
  vector operator *= (vector);
  vector operator *= (const nr_double_t);
  vector operator /= (vector);
  vector operator /= (const nr_double_t);

  // easy accessor operators
  nr_complex_t  operator () (int i) const { return data[i]; }

 private:
  int requested;
  int size;
  int capacity;
  strlist * dependencies;
  nr_complex_t * data;
};

#endif /* __VECTOR_H__ */
