/***************************************************************************
    copyright: (C) 2009 by Michael Margraf
 ***************************************************************************/

// -----------------------------------------------------------------------
// Converts text file formats.
// Input: File containing several rows with text numbers
// Output: Qucs data file
// -----------------------------------------------------------------------

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>

#include "parse_txt.h"
#include "config.h"
#include "complex.h"
#include "object.h"
#include "vector.h"
#include "strlist.h"
#include "dataset.h"

// ********************************************************************
// Reads the text file with name "filename" into memory and returns
// a pointer to the text if successful. Otherwise returns 0.
// The text memory needs to be released by "free(pointer)" when not
// used anymore.
char* readTextFile(char *filename)
{
  struct stat info;
  if(stat(filename, &info) != 0)
    return 0;

  FILE *pFile = fopen(filename, "r");
  if(pFile == 0)
    return 0;

  char *Buffer = (char*)malloc(info.st_size+1);
  if(Buffer)
    Buffer[(int)fread(Buffer, 1, info.st_size, pFile)] = 0;

  fclose(pFile);
  return Buffer;
}

// ********************************************************************
// global variables

#define MYBLOCKSIZE  1000
char *file, *pFile, **varNames;
double *buffer, *pBuffer;
int lineNo, buffer_size, numVars, nameIndex;


// ********************************************************************
// Checks if there are variable names in the first line.
bool checkForVariableNames()
{
  while(*pFile <= ' ') { // skip white-spaces
    if(*pFile == 0) {    // end of file ?
      fprintf(stderr, "ERROR: No data in file.");
      return false;
    }
    if(*pFile == '\n')
      lineNo++;
    pFile++;
  }

  nameIndex = 0;
  varNames = (char**)malloc(MYBLOCKSIZE * sizeof(char*));

  char c;
  while(true) {

    while(*pFile <= ' ') { // skip white-spaces
      if(*pFile == 0) {    // end of file ?
        fprintf(stderr, "ERROR: No data in file.");
        return false;
      }
      if(*pFile == '\n') { // end of line ?
        pFile++;
        lineNo++;
        return true;
      }
      pFile++;
    }

    c = (*pFile) | 0x20;
    if((c < 'a') || (c > 'z')) {
      if(nameIndex > 0) {
        fprintf(stderr, "ERROR: Wrong character in name line.");
        return false;
      }
      return true;
    }

    varNames[nameIndex++] = pFile;
    if(nameIndex >= MYBLOCKSIZE) {
      fprintf(stderr, "ERROR: Too many variables in name line.");
      return false;
    }

    while(*pFile > ' ') {
      c = *pFile;
      pFile++;
      if(c >= '0')
        if(c <= '9')
          continue;
      c |= 0x20;
      if(c >= 'a')
        if(c <= 'z')
          continue;
      *(pFile-1) = '_';   // replace all invalid characters
    }

    if(*pFile == 0) {    // end of file ?
      fprintf(stderr, "ERROR: No numbers in file.");
      return false;
    }

    *(pFile++) = 0;
  }
  return false;
}

// ********************************************************************
// Converts one line of characters into binary numbers and put them into "buffer".
// Returns the count of numbers in the line if successful.
// Returns -1 if end of file is reached and returns -10 if an error occured.
int convertLine()
{
  if(*pFile == 0)    // end of file ?
    return -1;

  char *pEnd;
  int count=0;
  while(true) {

    while(*pFile <= ' ') { // skip white-spaces
      if(*pFile == 0)      // end of file ?
        return count;
      if(*pFile == '\n') { // end of line ?
        pFile++;
        return count;
      }
      pFile++;
    }

    // enlarge buffer if necessary
    if((pBuffer-buffer) >= buffer_size) {
      buffer = (double*)realloc(buffer, buffer_size+MYBLOCKSIZE);
      pBuffer = buffer + buffer_size;
      buffer_size += MYBLOCKSIZE;
    }

    // convert to binary number
    *pBuffer = strtod(pFile, &pEnd);
    if(*pEnd > ' ') {
      fprintf(stderr, "ERROR: Wrong character encountered in line %d.", lineNo);
      return -10;
    }
    pFile = pEnd;
    pBuffer++;
	count++;
  }
  return 0;
}

// ********************************************************************
dataset* read_txt_file(char *filename, bool isComplex)
{
  // First, read file.
  file = readTextFile(filename);
  if(file == 0) {
    fprintf(stderr, "ERROR: Cannot read file '%s'.", filename);
    return 0;
  }

  lineNo = 0;
  pFile = file;

  if(!checkForVariableNames())
    return 0;

  buffer_size = MYBLOCKSIZE;
  buffer = pBuffer = (double*)malloc(buffer_size);

  // Count numbers in first line.
  // Each line must have the same count of numbers.
  int result;
  if(nameIndex == 0)
    do {
      lineNo++;
      result = convertLine();
      if(result == -1) {
        fprintf(stderr, "ERROR: No data in file '%s'.", filename);
        return 0;
      }
      if(result < 0)  // error?
        return 0;
    } while(result < 1);
  else
    result = nameIndex;

  numVars = result;  // count of numbers in one text line
  if(isComplex)
    if((numVars & 1) == 0) {
      fprintf(stderr, "ERROR: Number of data columns is even, but must be odd for complex data.");
      return 0;
    }

  // Now, convert the whole file into binary numbers.
  while(true) {
    do {
      lineNo++;
      result = convertLine();
    } while(result == 0);  // omit empty lines
    if(result == -1)  // end of file?
	   break;
    if(result < 0)  // error?
      return 0;
    if(result != numVars) {
      fprintf(stderr, "ERROR: Each line must contain %d numbers, in line %d are %d.",
              numVars, lineNo, result);
      return 0;
    }
  }

  buffer_size = pBuffer - buffer;  // count of numbers


  // Now create qucs dataset. ---------------------------------------
  vector *v;
  int i, j, k;

  dataset *dataset_result = new dataset();

  strlist *sl = new strlist();
  if(nameIndex > 0)
    sl->append(varNames[0]);
  else
    sl->append("indepVar");

  // first create independent variable
  v = new vector(sl->get(0), buffer_size/numVars);
  for(i=j=0; j<buffer_size; j+=numVars)
    v->set(buffer[j], i++);
  dataset_result->appendDependency(v);

  // create dependent variables
  char vname[24];
  for(i=1; i<numVars; i++) {
    if(nameIndex > 0)
      v = new vector(varNames[i], buffer_size/numVars);
    else {
      sprintf(vname, "depVar%d", i);
      v = new vector(vname, buffer_size/numVars);
    }
    v->setDependencies(sl);

    k = 0;
    pBuffer = buffer + i;
    for(j=0; j<buffer_size; j+=numVars)
      if(isComplex)
        v->set(nr_complex_t(pBuffer[j], pBuffer[j+1]), k++);
      else
        v->set(nr_complex_t(pBuffer[j], 0.0), k++);

    dataset_result->appendVariable(v);
    if(isComplex)
      i++;
  }

  free(file);  // contains strings "varNames"
  free(buffer);
  free(varNames);
  return dataset_result;
}
