/*
 * dataset.cpp - dataset class implementation
 *
 * Copyright (C) 2003, 2004, 2005, 2006, 2007 Stefan Jahn <stefan@lkcc.org>
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this package; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street - Fifth Floor,
 * Boston, MA 02110-1301, USA.  
 *
 * $Id: dataset.cpp,v 1.26 2008/02/15 17:55:59 ela Exp $
 *
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <assert.h>
#include <math.h>

#include "logging.h"
#include "complex.h"
#include "object.h"
#include "strlist.h"
#include "vector.h"
#include "dataset.h"
#include "check_touchstone.h"
#include "check_csv.h"
#include "check_citi.h"
#include "check_zvr.h"
#include "check_mdl.h"

// Constructor creates an unnamed instance of the dataset class.
dataset::dataset () : object () {
  variables = dependencies = NULL;
  file = NULL;
}

// Destructor deletes a dataset object.
dataset::~dataset () {
  vector * n, * v;
  // delete dependency vectors
  for (v = dependencies; v != NULL; v = n) {
    n = (vector *) v->getNext ();
    delete v;
  }
  // delete variable vectors
  for (v = variables; v != NULL; v = n) {
    n = (vector *) v->getNext ();
    delete v;
  }
  if (file) free (file);
}

// This function adds a dependency vector to the current dataset.
void dataset::addDependency (vector * v) {
  if (dependencies) dependencies->setPrev (v);
  v->setNext (dependencies);
  v->setPrev (NULL);
  dependencies = v;
}

// This function removes a dependency vector from the current dataset.
void dataset::delDependency (vector * v) {
  if (dependencies == v) {
    dependencies = (vector *) v->getNext ();
    if (dependencies) dependencies->setPrev (NULL);
  }
  else {
    vector * next = (vector *) v->getNext ();
    vector * prev = (vector *) v->getPrev ();
    prev->setNext (next);
    if (next) next->setPrev (prev);
  }
  delete v;
}

// This function appends a dependency vector to the current dataset.
void dataset::appendDependency (vector * v) {
  vector * e;
  if (dependencies) {
    for (e = dependencies; e->getNext (); e = (vector *) e->getNext ()) ;
    v->setPrev (e);
    e->setNext (v);
  }
  else {
    v->setPrev (NULL);
     dependencies= v;
  }
  v->setNext (NULL);
}

// This function adds a variable vector to the current dataset.
void dataset::addVariable (vector * v) {
  if (variables) variables->setPrev (v);
  v->setNext (variables);
  v->setPrev (NULL);
  variables = v;
}

// This function removes a variable vector from the current dataset.
void dataset::delVariable (vector * v) {
  if (variables == v) {
    variables = (vector *) v->getNext ();
    if (variables) variables->setPrev (NULL);
  }
  else {
    vector * next = (vector *) v->getNext ();
    vector * prev = (vector *) v->getPrev ();
    prev->setNext (next);
    if (next) next->setPrev (prev);
  }
  delete v;
}

// This function appends a variable vector to the current dataset.
void dataset::appendVariable (vector * v) {
  vector * e;
  if (variables) {
    for (e = variables; e->getNext (); e = (vector *) e->getNext ()) ;
    v->setPrev (e);
    e->setNext (v);
  }
  else {
    v->setPrev (NULL);
    variables = v;
  }
  v->setNext (NULL);
}

/* The function goes through the list of dependencies in the dataset
   and returns the vector specified by the given name.  Otherwise the
   function returns NULL. */
vector * dataset::findDependency (const char * n) {
  for (vector * v = dependencies; v != NULL; v = (vector *) v->getNext ()) {
    if (!strcmp (v->getName (), n))
      return v;
  }
  return NULL;
}

/* The function goes through the list of variables in the dataset and
   returns the vector specified by the given name.  If there is no
   such variable registered the function returns NULL. */
vector * dataset::findVariable (const char * n) {
  for (vector * v = variables; v != NULL; v = (vector *) v->getNext ()) {
    if (!strcmp (v->getName (), n))
      return v;
  }
  return NULL;
}

// Returns the number of variable vectors.
int dataset::countVariables (void) {
  int count = 0;
  for (vector * v = variables; v != NULL; v = (vector *) v->getNext ())
    count++;
  return count;
}

// Returns the number of dependency vectors.
int dataset::countDependencies (void) {
  int count = 0;
  for (vector * v = dependencies; v != NULL; v = (vector *) v->getNext ())
    count++;
  return count;
}

/* Sets the current output file name.  The file name is used during
   the print functionality of the dataset class. */
void dataset::setFile (const char * f) {
  if (file)
    free (file);
  file = f ? _strdup (f) : NULL;
}


/* This function prints the current dataset representation either to
   the specified file name (given by the function setFile()) or to
   stdout if there is no such file name given. */
void dataset::print (void) {

  FILE * f = stdout;

  // open file for writing
  if (file) {
    if ((f = fopen (file, "wb")) == NULL) {
      logprint (LOG_ERROR, "cannot create file `%s': %s\n",
                file, strerror (errno));
      return;
    }
  }

  // print prologue
  int n;
  fputs (DATA_FORMAT_ID, f);
  n = QUCS_HEX_VERSION;
  fwrite(&n, sizeof(int), 1, f);

  int count = 0;   // count number of bytes for header
  fwrite(&count, sizeof(int), 1, f);   // size of header (placeholder)

  // print dependencies in header
  for (vector * d = dependencies; d != NULL; d = (vector *) d->getNext ()) {
    count += printDependency (d, f);
  }

  // print variables in header
  for (vector * v = variables; v != NULL; v = (vector *) v->getNext ()) {
    if (v->getDependencies () != NULL)
      count += printVariable (v, f);
    else
      count += printDependency (v, f);
  }

  // print dependencies
  char *varName;
  for (vector * d = dependencies; d != NULL; d = (vector *) d->getNext ()) {
    varName = d->getName ();
    if(*varName == '_')
      continue;

    d->saveData (f, true);
  }

  // print variables
  for (vector * v = variables; v != NULL; v = (vector *) v->getNext ()) {
    varName = v->getName ();
    if(*varName == '_')
      continue;

    if (v->getDependencies () != NULL)
      v->saveData (f);
    else
      v->saveData (f, true);
  }

  fseek(f, 12, SEEK_SET);
  fwrite(&count, sizeof(int), 1, f);   // size of header

  // close file if necessary
  if (file) fclose (f);
}

/* Prints the given vector's header as independent dataset vector into the
   given file descriptor. */
int dataset::printDependency (vector * v, FILE * f)
{
  char *varName = v->getName();
  if(*varName == '_')  // don't print variable if not named by the user
    return 0;

  int n = VARTYPE_IS_VAR + VARTYPE_IS_REAL;
  fwrite(&n, sizeof(int), 1, f);
  n = v->getSize ();
  fwrite(&n, sizeof(int), 1, f);
  fputs (varName, f);
  fputc (0, f);

  return strlen(varName) + 9;  // +9 -> type (4Bytes), size (4Bytes), null-termination (1Byte)
}

/* Prints the given vector's header as dependent dataset vector into the given
   file descriptor. */
int dataset::printVariable (vector * v, FILE * f)
{
  char *varName = v->getName();
  if(*varName == '_')  // don't print variable if not named by the user
    return 0;

  int count =  v->getSize ();
  int n = VARTYPE_IS_VAR + VARTYPE_IS_COMPLEX;
  if(count > 1)
    n += VARTYPE_IS_DEP;
  fwrite(&n, sizeof(int), 1, f);
  fwrite(&count, sizeof(int), 1, f);
  fputs (varName, f);

  vector * vdep;
  count = 0;
  if (v->getDependencies () != NULL) {
    for (strlistiterator it (v->getDependencies ()); *it; ++it) {
      vdep = findDependency (*it);
      if (vdep)
        if (vdep->getSize() <= 1)   // don't show if size is 1
          continue;
      if(count == 0)
        fputc (0, f);
      else
        fputc (' ', f);
      fputs (*it, f);
      count += strlen(*it) + 1;
    }
  }

  fputc (0, f);

  return count + strlen(varName) + 9;  // +9 -> type (4Bytes), size (4Bytes), null-termination (1Byte)
}

/* This static function read a full dataset from the given file and
   returns it.  On failure the function emits appropriate error
   messages and returns NULL. */
dataset * dataset::load (const char * file)
{
  FILE * f;
  f = fopen (file, "rb");
  if(f == NULL) {
    logprint (LOG_ERROR, "error loading `%s': %s\n", file, strerror (errno));
    return NULL;
  }

  char prologue[SIZE_PROLOGUE];
  if(fread(prologue, 1, SIZE_PROLOGUE, f) != SIZE_PROLOGUE) {  // read prologue
    fclose(f);
    logprint (LOG_ERROR, "error loading prologue of `%s': %s\n", file, strerror (errno));
    return NULL;
  }

  if(strcmp(prologue, DATA_FORMAT_ID) != 0) {  // Is it new data format?
    fclose(f);
    logprint (LOG_ERROR, "no Qucs data format in `%s'.\n", file);
    return NULL;
  }

  int size = *((int*)(prologue+12));
  char *header = (char*)malloc(size);
  if((int)fread(header, 1, size, f) != size) {  // read header
    fclose(f);
    free(header);
    logprint (LOG_ERROR, "error loading header of `%s': %s\n", file, strerror (errno));
    return NULL;
  }

  dataset *dataset_result = new dataset();

  vector *v;
  char varType;
  char *p = header;
  char *pEnd = header + size;
  while(p < pEnd) {  // collect all variables in header
    varType = *p;
    p += 8;
    if(varType & VARTYPE_IS_DIGI)
      break;   // digital variables are not supported

    size = *((int*)(p-4));
    v = new vector(p, size); // vector with name and size
    p += strlen(p) + 1;      // jump over variable name
    if(varType & VARTYPE_IS_DEP) {   // dependend variable?
      strlist *sl = new strlist();
      char *pe;
      do {
        pe = strchr(p, ' ');
        if(pe)
          *pe = 0;
        sl->append(p);
        if(pe)
          p = pe + 1;
      } while(pe);
      p += strlen(p) + 1;  // jump over last dependency
      v->setDependencies (sl);
      v->loadData(f, (varType & VARTYPE_IS_REAL) != 0);
      dataset_result->appendVariable (v);
    }
    else {
      v->loadData(f, true);
      dataset_result->appendDependency (v);
    }
  }
  free(header);

  fclose (f);
  dataset_result->setFile (file);
  return dataset_result;
}

/* This static function read a full dataset from the given touchstone
   file and returns it.  On failure the function emits appropriate
   error messages and returns NULL. */
dataset * dataset::load_touchstone (const char * file) {
  FILE * f;
  if ((f = fopen (file, "r")) == NULL) {
    logprint (LOG_ERROR, "error loading `%s': %s\n", file, strerror (errno));
    return NULL;
  }
  touchstone_in = f;
  touchstone_restart (touchstone_in);
  if (touchstone_parse () != 0) {
    fclose (f);
    return NULL;
  }
  if (touchstone_check () != 0) {
    fclose (f);
    return NULL;
  }
  fclose (f);
  touchstone_lex_destroy ();
  touchstone_result->setFile (file);
  return touchstone_result;
}

/* This static function read a full dataset from the given CSV file
   and returns it.  On failure the function emits appropriate error
   messages and returns NULL. */
dataset * dataset::load_csv (const char * file) {
  FILE * f;
  if ((f = fopen (file, "r")) == NULL) {
    logprint (LOG_ERROR, "error loading `%s': %s\n", file, strerror (errno));
    return NULL;
  }
  csv_in = f;
  csv_restart (csv_in);
  if (csv_parse () != 0) {
    fclose (f);
    return NULL;
  }
  if (csv_check () != 0) {
    fclose (f);
    return NULL;
  }
  fclose (f);
  csv_lex_destroy ();
  csv_result->setFile (file);
  return csv_result;
}

/* The function read a full dataset from the given CITIfile and
   returns it.  On failure the function emits appropriate error
   messages and returns NULL. */
dataset * dataset::load_citi (const char * file) {
  FILE * f;
  if ((f = fopen (file, "r")) == NULL) {
    logprint (LOG_ERROR, "error loading `%s': %s\n", file, strerror (errno));
    return NULL;
  }
  citi_in = f;
  citi_restart (citi_in);
  if (citi_parse () != 0) {
    fclose (f);
    return NULL;
  }
  if (citi_check () != 0) {
    fclose (f);
    return NULL;
  }
  fclose (f);
  citi_lex_destroy ();
  citi_result->setFile (file);
  return citi_result;
}

/* The function read a full dataset from the given ZVR file and
   returns it.  On failure the function emits appropriate error
   messages and returns NULL. */
dataset * dataset::load_zvr (const char * file) {
  FILE * f;
  if ((f = fopen (file, "r")) == NULL) {
    logprint (LOG_ERROR, "error loading `%s': %s\n", file, strerror (errno));
    return NULL;
  }
  zvr_in = f;
  zvr_restart (zvr_in);
  if (zvr_parse () != 0) {
    fclose (f);
    return NULL;
  }
  if (zvr_check () != 0) {
    fclose (f);
    return NULL;
  }
  fclose (f);
  zvr_lex_destroy ();
  if (zvr_result) zvr_result->setFile (file);
  return zvr_result;
}

/* The function read a full dataset from the given MDL file and
   returns it.  On failure the function emits appropriate error
   messages and returns NULL. */
dataset * dataset::load_mdl (const char * file) {
  FILE * f;
  if ((f = fopen (file, "r")) == NULL) {
    logprint (LOG_ERROR, "error loading `%s': %s\n", file, strerror (errno));
    return NULL;
  }
  mdl_in = f;
  mdl_restart (mdl_in);
  if (mdl_parse () != 0) {
    fclose (f);
    return NULL;
  }
  if (mdl_check () != 0) {
    fclose (f);
    return NULL;
  }
  fclose (f);
  mdl_lex_destroy ();
  if (mdl_result) mdl_result->setFile (file);
  return mdl_result;
}
