/*
 * touchstone_producer.cpp - the Touchstone data file producer
 *
 * Copyright (C) 2007 Stefan Jahn <stefan@lkcc.org>
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this package; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street - Fifth Floor,
 * Boston, MA 02110-1301, USA.  
 *
 * $Id: touchstone_producer.cpp,v 1.5 2008/01/13 10:50:22 ela Exp $
 *
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <string.h>

#include "touchstone_producer.h"
#include "logging.h"
#include "constants.h"

/* Global variables. */
/* dataset * qucs_data = NULL;   -- already defined in CSV producer */
/* FILE * touchstone_out = NULL; -- already defined in Touchstone lexer */

struct touchstone_data_t {
  char parameter;      // type of variable
  int ports;           // number of S-parameter ports
  double resistance;   // reference impedance
  const char * format; // data format
  vector * vd;         // appropriate dependency vector
  vector * mv;         // appropriate data vector
  vector * fmin;       // minimum noise figure
  vector * sopt;       // optimum input refelction for minimum noise figure
  vector * rn;         // effective noise resistance
  vector * vf;         // dependency vector for noise
}
touchstone_data;

/* Definition of line separator. */
//#define touchstone_crlf "\r\n"
// VS2008 creates the "\r" automatically! So "\n" only is the right choice.
#define touchstone_crlf "\n"

/* opens the given file, fallback to stdin/stdout */
FILE * open_file (char * file, const char * flag) {
  FILE * fd = NULL;
  if (file) {
    if ((fd = fopen (file, flag)) == NULL) {
      fprintf (stderr, "cannot open file `%s': %s\n", file, strerror (errno));
      return 0;
    }
  }
  else {
    fd = flag[0] == 'r' ? stdin : stdout;
  }
  return fd;
}

/* The Touchstone noise data printer. */
void touchstone_print_noise()
{
  if (touchstone_data.vf != NULL && touchstone_data.sopt != NULL &&
      touchstone_data.rn != NULL && touchstone_data.fmin != NULL) {
    // blank line separator
    fprintf (touchstone_out, touchstone_crlf);
    // noise data
    for(int i = 0; i < touchstone_data.vf->getSize(); i++) {
      nr_double_t f = real(touchstone_data.vf->get(i));
      fprintf (touchstone_out, "%." NR_DECS "e"
	       " %+." NR_DECS "e"
	       " %+." NR_DECS "e" " %+." NR_DECS "e"
	       " %+." NR_DECS "e"
	       touchstone_crlf, f,
	       real (touchstone_data.fmin->get (i)),
	       abs (touchstone_data.sopt->get (i)),
	       deg (arg (touchstone_data.sopt->get (i))),
	       real (touchstone_data.rn->get (i)) / touchstone_data.resistance);
    }
  }
}

/* The Touchstone data printer. */
void touchstone_print()
{
  nr_double_t f;
  nr_complex_t S;
  int i, c, r, ports, size;

  // header line
  fprintf (touchstone_out, "# %s %c %s R %g" touchstone_crlf,
	   "HZ", touchstone_data.parameter, touchstone_data.format,
	   touchstone_data.resistance);

  ports = touchstone_data.ports;
  size = touchstone_data.vd->getSize();

  // one-port file
  if(ports == 1) {
    for(i = 0; i < size; i++) {
      S = touchstone_data.mv->get(i);
      f = real (touchstone_data.vd->get (i));
      fprintf (touchstone_out, "%." NR_DECS "e"
	       " %+." NR_DECS "e" " %+." NR_DECS "e"
	       touchstone_crlf, f, real(S), imag(S));
    }    
  }
  // two-port file (S12 and S21 are exchanged, thus special treatment)
  else if(ports == 2) {
    for(i = 0; i < size; i++) {
      f = real(touchstone_data.vd->get(i));
      fprintf(touchstone_out, "%." NR_DECS "e", f);
      S = touchstone_data.mv->get((0+2*0)*size + i);
      fprintf(touchstone_out, " %+." NR_DECS "e" " %+." NR_DECS "e",
              real(S), imag(S));
      S = touchstone_data.mv->get((0+2*1)*size + i);
      fprintf(touchstone_out, " %+." NR_DECS "e" " %+." NR_DECS "e",
              real(S), imag(S));
      S = touchstone_data.mv->get((1+2*0)*size + i);
      fprintf(touchstone_out, " %+." NR_DECS "e" " %+." NR_DECS "e",
              real(S), imag(S));
      S = touchstone_data.mv->get((1+2*1)*size + i);
      fprintf(touchstone_out, " %+." NR_DECS "e" " %+." NR_DECS "e",
              real(S), imag(S));
      fprintf (touchstone_out, touchstone_crlf);
    }
  }
  // four-port and above files
  else if(touchstone_data.ports > 2) {
    for(i = 0; i < size; i++) {
      f = real(touchstone_data.vd->get(i));
      fprintf(touchstone_out, "%." NR_DECS "e", f);
      for(r = 0; r < ports; r++) {
        if(r > 0)
          fprintf (touchstone_out, "      %"  NR_DECS "s", " ");
        for(c = 0; c < ports; c++) {
          if(c > 0 && (c & 3) == 0)  // reached maximum count per line?
            fprintf(touchstone_out, touchstone_crlf "      %"  NR_DECS "s", " ");
          S = touchstone_data.mv->get((c+ports*r)*size + i);
          fprintf(touchstone_out, " %+." NR_DECS "e" " %+." NR_DECS "e",
                  real(S), imag(S));
        }
        fprintf (touchstone_out, touchstone_crlf);
      }
    }
  }
}

/* The function finds an appropriate S-parameters or other parameter
   type (G, H, Y, Z, etc.) matrix from the given dataset and stores it
   into the global Touchstone structure. */
void touchstone_find_data(dataset *data, const char *name)
{
  vector *v;
  int rs, cs;
  strlist *deps;
  char *n, *vn, *vd = NULL, *vf = NULL;

  // find parameter matrix data and its dimensions
  for (v = data->getVariables (); v != NULL; v = (vector *) v->getNext ()) {
    vn = v->getName ();
    // requested matrix vector name found?
    if (strcmp (vn, name) == 0) {
      if ((deps = v->getDependencies ()) != NULL) {
        touchstone_data.mv = v;
        vd = deps->get(0);
        rs = strtoul(deps->get(1), &n, 10);
        cs = strtoul(deps->get(2), &n, 10);
        touchstone_data.ports = rs > cs ? rs : cs;
      }
    }
    // minimum noise figure?
    if (!strcmp (vn, "NFmin")) {
      touchstone_data.fmin = v;
      if ((deps = v->getDependencies ()) != NULL)
        vf = deps->get (0);
    }
    // optimal input reflection for minimum noise figure?
    else if (!strcmp (vn, "Sopt")) {
      touchstone_data.sopt = v;
      if ((deps = v->getDependencies ()) != NULL)
        vf = deps->get (0);
    }
    // effective noise resitance?
    else if (!strcmp (vn, "Rn")) {
      touchstone_data.rn = v;
      if ((deps = v->getDependencies ()) != NULL)
        vf = deps->get (0);
    }
  }

  // look for dependency (frequency) vector
  for(v = data->getDependencies(); v != NULL; v = (vector*)v->getNext()) {
    if(vd && !strcmp(v->getName(), vd))
      touchstone_data.vd = v;
    if(vf && !strcmp(v->getName(), vf))
      touchstone_data.vf = v;
  }
}

/* This is the overall Touchstone producer. */
int touchstone_producer (const char * variable, char * outfile) {

  // apply data variable name
  if (variable == NULL) {
    variable = "S";
  }

  // initialize global Touchstone structure
  touchstone_data.mv = NULL;
  touchstone_data.vd = NULL;
  touchstone_data.format = "RI";
  touchstone_data.resistance = 50.0;
  touchstone_data.fmin = NULL;
  touchstone_data.sopt = NULL;
  touchstone_data.rn = NULL;
  touchstone_data.vf = NULL;
  touchstone_data.parameter = toupper(variable[0]);

  // look for appropriate matrix data
  touchstone_find_data (qucs_data, variable);

  // print matrix data if available
  if (touchstone_data.mv == NULL) {
    fprintf (stderr, "no s-parameters found in input file\n");
    return -1;
  }
  if (touchstone_data.vd == NULL) {
    fprintf (stderr, "no dependency found in input file\n");
    return -1;
  }

  if ((touchstone_out = open_file (outfile, "w")) == NULL)
    return -1;
  touchstone_print ();
  touchstone_print_noise ();
  fclose (touchstone_out);
  return 0;
}
