
# ifndef  _QF_TFORM_H
# define  _QF_TFORM_H

#include <QTextStream>

// Headers for standard transformations
// Those transformations are : lowpass (no transform), highpass
// standard bandpass (geometrical type) and bandstop (inverse bandpass)

extern	struct	qf_filter_api* qf_filter_apis [];
extern	struct	qf_tform_api*	qf_tform_apis [];

// Virtual holder
class qf_tform {
  protected:
  friend	      struct	qf_ind;	      // To access Tspec in denorm
  friend	      struct	qf_cap;	      // To access Tspec in denorm
  friend	      struct	qf_plc;	      // To access Tspec in denorm
  friend	      struct	qf_slc;	      // To access Tspec in denorm

  QString     name;
  qf_spec*	      Tspec;
  qf_lcmp	      lcmp;
  virtual qf_double_t th	(void) = 0;

		      qf_tform	(qf_spec* Ts) : Tspec (Ts) {};
  public:

  void	      dump	(QTextStream&);
};

// A common static function to dispatch denormalization

void		     denorm	(qf_tform*, qf_cmp*);

// API for transformation

struct	qf_tform_api {
  QString     name;
  qf_tform*   (* cons)	(qf_spec*);		    // Constructor
};

// Lowpass
class qf_lowpass : public qf_tform {
  protected:
		  qf_lowpass  (qf_spec* Ts);
      qf_double_t th	      (void)	    {return Tspec -> fs / Tspec -> fc;}

  public:

friend qf_tform*  lowpass     (qf_spec* Tspec) {return new qf_lowpass (Tspec);}
};

qf_tform*	  lowpass     (qf_spec*);


// Highpass
class qf_highpass : public qf_tform {
  private:
		  qf_highpass  (qf_spec* Ts);
      qf_double_t th	       (void)	  {return Tspec -> fc / Tspec -> fs;}

  public:

friend qf_tform*  highpass     (qf_spec* spec) {return new qf_highpass (spec);}
};

qf_tform*	  highpass     (qf_spec*);

// Bandpass
class qf_bandpass : public qf_tform {
  private:
		  qf_bandpass  (qf_spec* s);
      qf_double_t th	       (void)	    {return 
					     fabs (Tspec -> fs -
						   Tspec -> fc * Tspec -> fc /
						   Tspec -> fs) / Tspec -> bw;}
  public:

friend qf_tform*  bandpass     (qf_spec* spec) {return new qf_bandpass (spec);}
};

qf_tform*	  bandpass     (qf_spec*);

// Bandstop
class qf_bandstop: public qf_tform {
  private:
		  qf_bandstop  (qf_spec* s);
      qf_double_t th	       (void)	    {qf_double_t 
					      sh = Tspec -> fs / Tspec -> fc;
					      qf_double_t	
					       bw = Tspec -> bw / Tspec -> fc;
					       Tspec -> bw = 
						fabs (sh - 1/sh) * Tspec -> fc;
					       return 1/fabs (
						sh * bw / (sh * sh - 1));}

  public:

friend qf_tform*  bandstop     (qf_spec* spec) {return new qf_bandstop (spec);}
};

qf_tform*	  bandstop     (qf_spec*);

 # ifdef  _QF_API

struct	qf_tform_api	  qf_lowpass_api = {
  "Lowpass",
  lowpass
};

struct	qf_tform_api	  qf_highpass_api = {
  "Highpass",
  highpass
};

struct	qf_tform_api	  qf_bandpass_api = {
  "Bandpass",
  bandpass
};

struct	qf_tform_api	  qf_bandstop_api = {
  "Bandstop",
  bandstop
};

  # endif // _QF_API
# endif	  // _QF_TFORM_H
