/***************************************************************************
    copyright: (C) 2005 by Michael Margraf
 ***************************************************************************/

#include "config.h"

#include <QDir>
#include <QFile>
#include <QMenu>
#include <QLabel>
#include <QTimer>
#include <QMenuBar>
#include <QMimeData>
#include <QGroupBox>
#include <QLineEdit>
#include <QCheckBox>
#include <QComboBox>
#include <QClipboard>
#include <QVBoxLayout>
#include <QGridLayout>
#include <QMessageBox>
#include <QPushButton>
#include <QTextStream>
#include <QRadioButton>
#include <QApplication>

#include "lc_filter.h"
#include "tl_filter.h"
#include "eqn_filter.h"
#include "lc_diplexer.h"
#include "line_filter.h"
#include "stub_filter.h"
#include "cline_filter.h"
#include "stepz_filter.h"
#include "active_filter.h"
#include "coupledlc_filter.h"

#include "qf_poly.h"
#include "qf_comp.h"
#include "qf_tform.h"
#include "qf_filter.h"
#include "qucsfilter.h"
#include "mywidgets.h"


QucsFilter::QucsFilter() : QDialog(0, Qt::Dialog | Qt::WindowMinimizeButtonHint)
{
  setWindowTitle(tr("QucsStudio Filter Synthesis ") + PACKAGE_VERSION);
  QDoubleValidator *doubleVal = new QDoubleValidator(this);
  doubleVal->setBottom(0.0);

  QVBoxLayout *all = new QVBoxLayout(this);
  all->setSpacing(2);
  all->setMargin(1);

  // --------  create menubar  -------------------
  QMenu *fileMenu = new QMenu(tr("&File"));
  fileMenu->addAction(tr("E&xit"), this, SLOT(accept()), Qt::CTRL+Qt::Key_Q);

  QMenu *helpMenu = new QMenu(tr("&Help"));
  helpMenu->addAction(tr("Help..."), this, SLOT(slotHelpIntro()), Qt::Key_F1);
  helpMenu->addSeparator();
  helpMenu->addAction(tr("&About..."), this, SLOT(slotHelpAbout()), 0);
  helpMenu->addAction(tr("About Qt..."), this, SLOT(slotHelpAboutQt()), 0);

  QMenuBar *bar = new QMenuBar;
  bar->addMenu(fileMenu);
  bar->addMenu(helpMenu);
  all->addWidget(bar);

  // -------  create main windows widgets --------
  QHBoxLayout *h1 = new QHBoxLayout;
  all->addLayout(h1);
  h1->setSpacing(2);
  h1->setMargin(5);

  // ...........................................................
  QGroupBox *boxFilter = new QGroupBox(tr("Filter"));
  h1->addWidget(boxFilter);

  QGridLayout *gbox = new QGridLayout;
  gbox->setSpacing(2);
  gbox->setMargin(3);

  gbox->addWidget(new QLabel(tr("Realization:")), 0,0);
  ComboRealize = new QComboBox;
  ComboRealize->addItem(tr("LC ladder"));
  ComboRealize->addItem(tr("coupled LC resonators"));
  ComboRealize->addItem(tr("end-coupled transmission lines"));
  ComboRealize->addItem(tr("lateral-coupled transmission lines"));
  ComboRealize->addItem(tr("stepped-impedance"));
  ComboRealize->addItem(tr("stub transmission lines"));
  ComboRealize->addItem(tr("active"));
  ComboRealize->addItem(tr("equation-defined"));
  connect(ComboRealize, SIGNAL(activated(int)), SLOT(slotRealizationChanged(int)));
  gbox->addWidget(ComboRealize, 0,1, 1,2);

  gbox->addWidget(new QLabel(tr("Filter type:")), 1,0);
  ComboType = new QComboBox;
  ComboType->addItem("Bessel");
  ComboType->addItem("Butterworth");
  ComboType->addItem("Legendre");
  ComboType->addItem("Chebyshev");
  ComboType->addItem("inverse Chebyshev");
  ComboType->addItem("Cauer");
  connect(ComboType, SIGNAL(activated(int)), SLOT(slotTypeChanged(int)));
  gbox->addWidget(ComboType, 1,1, 1,2);

  gbox->addWidget(new QLabel(tr("Filter class:")), 2,0);
  ComboClass = new QComboBox;
  ComboClass->addItem(tr("Low pass"));
  ComboClass->addItem(tr("High pass"));
  ComboClass->addItem(tr("Band pass"));
  ComboClass->addItem(tr("Band stop"));
  ComboClass->addItem(tr("Diplexer"));
  ComboClass->addItem(tr("Triplexer"));
  connect(ComboClass, SIGNAL(activated(int)), SLOT(slotClassChanged(int)));
  gbox->addWidget(ComboClass, 2,1, 1,2);

  LabelOrder = new QLabel(tr("Order:"));
  gbox->addWidget(LabelOrder, 3,0);
  EditOrder = new QLineEdit("3");
  EditOrder->setValidator(new QIntValidator(1, 100, EditOrder));
  gbox->addWidget(EditOrder, 3,1);

  LabelStart = new QLabel(tr("Corner frequency:"));
  gbox->addWidget(LabelStart, 4,0);
  EditCorner = new QLineEdit("1");
  EditCorner->setValidator(doubleVal);
  gbox->addWidget(EditCorner, 4,1);
  ComboCorner = new UnitComboBox(QUANTITY_FREQUENCY);
  gbox->addWidget(ComboCorner, 4,2);

  LabelStop = new QLabel(tr("Stop frequency:"));
  gbox->addWidget(LabelStop, 5,0);
  EditStop = new QLineEdit("2");
  EditStop->setValidator(doubleVal);
  gbox->addWidget(EditStop, 5,1);
  ComboStop = new UnitComboBox(QUANTITY_FREQUENCY);
  gbox->addWidget(ComboStop, 5,2);

  LabelBandStop = new QLabel(tr("Stop band frequency:"));
  gbox->addWidget(LabelBandStop, 6,0);
  EditBandStop = new QLineEdit("3");
  EditBandStop->setValidator(doubleVal);
  gbox->addWidget(EditBandStop, 6,1);
  ComboBandStop = new UnitComboBox(QUANTITY_FREQUENCY);
  gbox->addWidget(ComboBandStop, 6,2);

  LabelRipple = new QLabel(tr("Pass band ripple:"));
  gbox->addWidget(LabelRipple, 7,0);
  EditRipple = new QLineEdit("1");
  EditRipple->setValidator(doubleVal);
  gbox->addWidget(EditRipple, 7,1);
  LabelRipple_dB = new QLabel("dB");
  gbox->addWidget(LabelRipple_dB, 7,2);

  LabelImpedance = new QLabel(tr("Impedance:"));
  gbox->addWidget(LabelImpedance, 8,0);
  EditImpedance = new QLineEdit("50");
  EditImpedance->setValidator(doubleVal);
  gbox->addWidget(EditImpedance, 8,1);
  LabelOhm = new QLabel("Ohm");
  gbox->addWidget(LabelOhm, 8,2);

  boxFilter->setLayout(gbox);

  // ...........................................................
  QVBoxLayout *v1 = new QVBoxLayout;
  v1->setSpacing(2);
  v1->setMargin(1);
  h1->addLayout(v1);

  boxTopology = new QGroupBox(tr("Topology"));
  v1->addWidget(boxTopology);

  QVBoxLayout *vt = new QVBoxLayout;
  vt->setSpacing(2);
  vt->setMargin(3);

  radioPi = new QRadioButton(tr("pi type"));
  radioPi->setChecked(true);
  vt->addWidget(radioPi);
  radioTee = new QRadioButton(tr("tee type"));
  vt->addWidget(radioTee);

  boxTopology->setLayout(vt);

  // ...........................................................
  boxSubstrate = new QGroupBox(tr("Microstrip Substrate"));
  v1->addWidget(boxSubstrate);

  QGridLayout *gbox2 = new QGridLayout;
  gbox2->setSpacing(2);
  gbox2->setMargin(3);

  checkMicrostrip = new QCheckBox(tr("use microstrip lines"));
  connect(checkMicrostrip, SIGNAL(stateChanged(int)), SLOT(slotSetMicrostrip(int)));
  gbox2->addWidget(checkMicrostrip, 0,0, 1,2);

  gbox2->addWidget(new QLabel(tr("relative permittivity:")), 1,0);
  ComboEr = new QuantityBox(List_er);
  gbox2->addWidget(ComboEr, 1,1, 1,2);

  gbox2->addWidget(new QLabel(tr("substrate height:")), 2,0);
  EditHeight = new QLineEdit("1.0");
  EditHeight->setValidator(doubleVal);
  gbox2->addWidget(EditHeight, 2,1);
  ComboHeight = new UnitComboBox();
  gbox2->addWidget(ComboHeight, 2,2);

  gbox2->addWidget(new QLabel(tr("metal thickness:")), 3,0);
  EditThickness = new QLineEdit("12.5");
  EditThickness->setValidator(doubleVal);
  gbox2->addWidget(EditThickness, 3,1);
  ComboThickness = new UnitComboBox();
  gbox2->addWidget(ComboThickness);

  LabelMin = new QLabel;
  gbox2->addWidget(LabelMin, 4,0);
  EditMinWidth = new QLineEdit;
  EditMinWidth->setValidator(doubleVal);
  gbox2->addWidget(EditMinWidth, 4,1);
  LabelMinUnit = new QLabel;
  gbox2->addWidget(LabelMinUnit, 4,2);

  LabelMax = new QLabel;
  gbox2->addWidget(LabelMax, 5,0);
  EditMaxWidth = new QLineEdit;
  EditMaxWidth->setValidator(doubleVal);
  gbox2->addWidget(EditMaxWidth, 5,1);
  LabelMaxUnit = new QLabel;
  gbox2->addWidget(LabelMaxUnit, 5,2);

  gbox2->addWidget(new QWidget, 6,0);
  gbox2->setRowStretch(6, 5);

  boxSubstrate->setLayout(gbox2);

  // ...........................................................
  all->addStretch(5);

  QPushButton *ButtonGo =
               new QPushButton(tr("Calculate and put into Clipboard"));
  connect(ButtonGo, SIGNAL(clicked()), SLOT(slotCalculate()));
  all->addWidget(ButtonGo, 0, Qt::AlignHCenter);

  LabelResult = new QLabel;
  ResultState = 100;
  slotShowResult();
  LabelResult->setAlignment(Qt::AlignHCenter);
  all->addWidget(LabelResult);

  // -------  finally set initial state  --------
  slotRealizationChanged(0);
  slotTypeChanged(0);
  slotClassChanged(0);
}

// -----------------------------------------------------------------------
void QucsFilter::slotHelpAbout()
{
  QMessageBox::about(this, tr("About..."),
    "QucsFilter Version " PACKAGE_VERSION "\n"+
    tr("Filter synthesis program") + "\n"
    "Copyright (C) 2005-2020  Michael Margraf\n" +
    "Copyright (C) 2005-2009  Vincent Habchi\n\n"+
    tr("This is free software. There is NO warranty at all,\nnot even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE."));
}

// -----------------------------------------------------------------------
void QucsFilter::slotHelpAboutQt()
{
  QMessageBox::aboutQt(this, tr("About Qt"));
}

// -----------------------------------------------------------------------
void QucsFilter::slotHelpIntro()
{
  QMessageBox::about(this, tr("Help..."),
    tr("QucsFilter is a filter synthesis program.\n"
       "To create a filter, simply enter all\n"
       "parameters and press the big button at the\n"
       "bottom of the main window. Immediatly, the\n"
       "schematic of the filter is calculated and\n"
       "put into the clipboard. Now go to the schematic\n"
       "editor, open an empty schematic and press\n"
       "CTRL-V (paste from clipboard). The filter\n"
       "schematic can now be inserted and\n"
       "simulated. Have lots of fun!") + "\n\n" +
    tr("filter types:\n"
       "Bessel - group delay as flat as possible\n"
       "Butterworth - frequency response as flat as possible\n"
       "Legendre - fastest roll-off without ripple\n"
       "Chebychev - constant ripple in pass band\n"
       "inverse Chebychev - constant ripple in stop band\n"
       "Cauer - constant ripple in pass and stop band") + "\n\n" +
    tr("Chebychev is the most popular typ of filter. Its\n"
       "ripple is usually chosen to be 0.5dB.\n"
       "Bessel was designed for broadband pulses like\n"
       "rectangular PRBS signals."));
}

// -----------------------------------------------------------------------
void QucsFilter::setError(const QString& Message)
{
  LabelResult->setText(tr("Result:") + "<font color=\"#FF0000\"><b>  " +
                       tr("Error") + "</b></font>");
  QMessageBox::critical(this, tr("Error"), Message);
}

// -----------------------------------------------------------------------
QByteArray QucsFilter::calculateFilter(struct tFilter *Filter)
{
  tSubstrate Substrate, *sub = 0;
  if(checkMicrostrip->isChecked()) {
    sub = &Substrate;
    Substrate.er = ComboEr->currentText().toDouble();
    Substrate.height = ComboHeight->value(EditHeight);
    Substrate.thickness = ComboThickness->value(EditThickness);
    Substrate.tand = 0.0;
    Substrate.resistivity = 2.4e-8;
    Substrate.minWidth = Filter->Zlow / 1e3;
    Substrate.maxWidth = Filter->Zhigh / 1e3;
  }

  switch(ComboRealize->currentIndex()) {
    case 1:  // C-coupled LC parallel resonance circuits
      return CoupledLC_Filter::createSchematic(Filter);
    case 2:  // C-coupled transmission line filter
       return Line_Filter::createSchematic(Filter, sub);
    case 3:  // coupled transmission line filter
       return CoupledLine_Filter::createSchematic(Filter, sub);
    case 4:  // stepped-impedance transmission line filter
      return StepImpedance_Filter::createSchematic(Filter, sub);
    case 5:  // quarter-wave stub transmission line filter
      return Stub_Filter::createSchematic(Filter, sub);
    case 6:  // active filter
      return Active_Filter::createSchematic(Filter, Filter->Impedance);
    case 7:  // equation defined filter
      return Equation_Filter::createSchematic(Filter);
  }


  // LC ladder filter
  if(Filter->Class == CLASS_DIPLEXER)
    return LC_Diplexer::createSchematic(Filter, false);
  if(Filter->Class == CLASS_TRIPLEXER)
    return LC_Diplexer::createSchematic(Filter, true);

  qf_spec spec;
  if(Filter->Type == TYPE_CAUER)
    spec.filter = 0;
  else if(Filter->Type == TYPE_INV_CHEBYSHEV)
    spec.filter = 1;
  else
    return LC_Filter::createSchematic(Filter);


  if((Filter->Class == CLASS_BANDPASS) || (Filter->Class == CLASS_BANDSTOP))
    spec.fc = sqrt(Filter->Frequency2 * Filter->Frequency) * M_TWO_PI;
  else
    spec.fc = Filter->Frequency * M_TWO_PI;
  spec.fs = Filter->Frequency3 * M_TWO_PI;
  spec.bw = fabs(Filter->Frequency2 - Filter->Frequency) * M_TWO_PI;
  spec.ord_given = true;
  spec.ord = Filter->Order;
  spec.subord = ' ';
  spec.amin = 10.0 * log10(Filter->Ripple);
  spec.amax = 0.0;
  spec.r1 = Filter->Impedance;
  spec.r2 = Filter->Impedance;

  spec.dual = radioTee->isChecked();
  spec.optc = false;

  spec.tform = Filter->Class;

  QString buf;
  QTextStream stream(&buf);
  qf_tform_apis[spec.tform]->cons(&spec)->dump(stream);

  return buf.toUtf8();
}

// -----------------------------------------------------------------------
void QucsFilter::slotCalculate()
{
  // get numerical values from input widgets
  double CornerFreq   = ComboCorner->value(EditCorner);
  double StopFreq     = ComboStop->value(EditStop);
  double BandStopFreq = ComboBandStop->value(EditBandStop);

  tFilter Filter;
  Filter.Type = ComboType->currentIndex();
  Filter.Class = ComboClass->currentIndex();
  Filter.Order = EditOrder->text().toInt();
  Filter.piType = radioPi->isChecked();
  Filter.Ripple = pow(10.0, EditRipple->text().toDouble()/10.0);
  Filter.Impedance = EditImpedance->text().toDouble();
  Filter.Frequency = CornerFreq;
  Filter.Frequency2 = StopFreq;
  Filter.Frequency3 = BandStopFreq;
  Filter.Zlow = EditMinWidth->text().toDouble();
  Filter.Zhigh = EditMaxWidth->text().toDouble();

  if(EditStop->isEnabled())
    if(Filter.Frequency >= Filter.Frequency2) {
      setError(tr("Stop frequency must be greater than start frequency."));
      return;
    }

  if(EditOrder->isEnabled()) {
    if(Filter.Order < 2) {
      setError(tr("Filter order must not be less than two."));
      return;
    }
  }

  QByteArray s = calculateFilter(&Filter);
  if(!s.isEmpty()) {
    // put resulting filter schematic into clipboard
    QMimeData *mime = new QMimeData;
    mime->setData(QUCSMIME, s);
    QApplication::clipboard()->setMimeData(mime);;

    LabelResult->setText(tr("Result:") + "<font color=\"#008000\"><b>  " +
                         tr("Successful") + "</b></font>");
  }
  else
    LabelResult->setText(tr("Result:") + "<font color=\"#FF0000\"><b>  " +
                         tr("Failed") + "</b></font>");

  // show result for some time
  QTimer::singleShot(4000, this, SLOT(slotShowResult()));
}

// -----------------------------------------------------------------------
void QucsFilter::slotShowResult()
{
  LabelResult->setText(tr("Result: --"));
}

// -----------------------------------------------------------------------
void QucsFilter::slotTypeChanged(int index)
{
  switch(index) {
    case TYPE_BESSEL:
    case TYPE_BUTTERWORTH:
    case TYPE_LEGENDRE:
      LabelRipple->setEnabled(false);
      EditRipple->setEnabled(false);
      LabelRipple_dB->setEnabled(false);
      break;
    case TYPE_CHEBYSHEV:
    case TYPE_INV_CHEBYSHEV:
    case TYPE_CAUER:
      LabelRipple->setEnabled(true);
      EditRipple->setEnabled(true);
      LabelRipple_dB->setEnabled(true);
      break;
  }
  if((index == TYPE_CAUER) || (index == TYPE_INV_CHEBYSHEV)){
    LabelBandStop->setEnabled(true);
    EditBandStop->setEnabled(true);
    ComboBandStop->setEnabled(true);
  }
  else {
    LabelBandStop->setEnabled(false);
    EditBandStop->setEnabled(false);
    ComboBandStop->setEnabled(false);
  }
}

// -----------------------------------------------------------------------
void QucsFilter::slotClassChanged(int index)
{
  switch(index) {
    case CLASS_LOWPASS:
    case CLASS_HIGHPASS:
    case CLASS_DIPLEXER:
      LabelStop->setEnabled(false);
      EditStop->setEnabled(false);
      ComboStop->setEnabled(false);
      LabelStart->setText(tr("Corner frequency:"));
      break;
    case CLASS_BANDPASS:
    case CLASS_BANDSTOP:
    case CLASS_TRIPLEXER:
      LabelStop->setEnabled(true);
      EditStop->setEnabled(true);
      ComboStop->setEnabled(true);
      LabelStart->setText(tr("Start frequency:"));
      break;
  }
  if(index == CLASS_BANDPASS) {
    LabelBandStop->setText(tr("Stop band frequency:"));
    LabelRipple->setText(tr("Pass band ripple:"));
  }
  else if(index == CLASS_BANDSTOP) {
    LabelBandStop->setText(tr("Pass band frequency:"));
    LabelRipple->setText(tr("Pass band attenuation:"));
  }
  else if(index == CLASS_TRIPLEXER) {
    LabelBandStop->setText(tr("Stop frequency:"));
    LabelRipple->setText(tr("Pass band ripple:"));
  }

  if((index == CLASS_DIPLEXER) || (index == CLASS_TRIPLEXER)) {
    ComboRealize->setCurrentIndex(0);  // LC ladder filter
    ComboRealize->setEnabled(false);
    boxTopology->setEnabled(false);
    radioTee->setChecked(true);
  }
  else {
    ComboRealize->setEnabled(true);
    index = ComboRealize->currentIndex();
    if((index == 0) || (index == 4) || (index == 5))
    boxTopology->setEnabled(true);
  }
}

// -----------------------------------------------------------------------
void QucsFilter::slotRealizationChanged(int index)
{
  ComboClass->setEnabled(true);
  if((index > 0) && (index < 4)) { // set to "transmission line" types?
    // set fixed to bandpass
    ComboClass->setCurrentIndex(CLASS_BANDPASS);
    slotClassChanged(CLASS_BANDPASS);
    ComboClass->setEnabled(false);
  }
  else if(index == 4) {  // set to "stepped impedance" types?
    // set fixed to lowpass
    ComboClass->setCurrentIndex(CLASS_LOWPASS);
    slotClassChanged(CLASS_LOWPASS);
    ComboClass->setEnabled(false);
  }
  else if(index == 5) {  // set to "stub line" types?
    ComboClass->setEnabled(true);
  }

  if(index == 6) {   // set to active filter?
    LabelImpedance->setText(tr("Voltage Gain:"));
    LabelOhm->setText(" ");
  }
  else {
    LabelImpedance->setText(tr("Impedance:"));
    LabelOhm->setText("ohms");
  }

  if(((index == 0) || (index == 4) || (index == 5))
     && (ComboClass->currentIndex() < CLASS_DIPLEXER))
    boxTopology->setEnabled(true);
  else
    boxTopology->setEnabled(false);

  if((index >= 2) && (index <= 5))
    boxSubstrate->setEnabled(true);
  else {
    checkMicrostrip->setChecked(false);
    slotSetMicrostrip(Qt::Unchecked);
    boxSubstrate->setEnabled(false);
  }
}

// -----------------------------------------------------------------------
void QucsFilter::slotSetMicrostrip(int state)
{
  bool enabled = (state == Qt::Checked);

  ComboEr->setEnabled(enabled);
  EditHeight->setEnabled(enabled);
  ComboHeight->setEnabled(enabled);
  EditThickness->setEnabled(enabled);
  ComboThickness->setEnabled(enabled);

  if(enabled) {
    LabelMin->setText(tr("minimum width:"));
    LabelMax->setText(tr("maximum width:"));
    LabelMinUnit->setText("mm");
    LabelMaxUnit->setText("mm");
    EditMinWidth->setText("0.4");
    EditMaxWidth->setText("8");
  }
  else {
    LabelMin->setText(tr("small impedance:"));
    LabelMax->setText(tr("big impedance:"));
    LabelMinUnit->setText("ohms");
    LabelMaxUnit->setText("ohms");
    EditMinWidth->setText("20");
    EditMaxWidth->setText("400");
  }
}

// -----------------------------------------------------------------------
// Loads the settings file and stores the settings.
bool QucsFilter::loadSettings(QDir *homeDir)
{
  bool result = true;
  QString Line, Setting;

  QFile file(homeDir->absoluteFilePath("filterrc"));
  if(!file.open(QIODevice::ReadOnly))
    result = false; // settings file doesn't exist
  else {
    QTextStream stream(&file);
    while(!stream.atEnd()) {
      Line = stream.readLine();
      Setting = Line.section('=',0,0);
      Line = Line.section('=',1,1);
      if(Setting == "FilterWindow")
        move(Line.section(",",0,0).toInt(), Line.section(",",1,1).toInt());
      else if(Setting == "Combo_Class") {
        ComboClass->setCurrentIndex(Line.toInt());
        slotClassChanged(Line.toInt());
      }
      else if(Setting == "Combo_Type") {
        ComboType->setCurrentIndex(Line.toInt());
        slotTypeChanged(Line.toInt());
      }
      else if(Setting == "Combo_Realize")
        ComboRealize->setCurrentIndex(Line.toInt());
      else if(Setting == "Edit_Order")
        EditOrder->setText(Line);
      else if(Setting == "Edit_Corner")
        EditCorner->setText(Line);
      else if(Setting == "Combo_Corner")
        ComboCorner->setCurrentIndex(Line.toInt());
      else if(Setting == "Edit_Stop")
        EditStop->setText(Line);
      else if(Setting == "Combo_Stop")
        ComboStop->setCurrentIndex(Line.toInt());
      else if(Setting == "Edit_BandStop")
        EditBandStop->setText(Line);
      else if(Setting == "Combo_BandStop")
        ComboBandStop->setCurrentIndex(Line.toInt());
      else if(Setting == "Edit_Ripple")
        EditRipple->setText(Line);
      else if(Setting == "Edit_Impedance")
        EditImpedance->setText(Line);
      else if(Setting == "Radio_Topology")
        radioTee->setChecked(Line == "1");
      else if(Setting == "Check_Microstrip")
        checkMicrostrip->setChecked(Line == "1");
      else if(Setting == "Combo_Er")
        ComboEr->setEditText(Line);
      else if(Setting == "Edit_Thickness") {
        EditThickness->setText(Line.section(",",0,0));
        ComboThickness->setCurrentIndex(Line.section(",",1,1).toUInt());
      }
      else if(Setting == "Edit_Height") {
        EditHeight->setText(Line.section(",",0,0));
        ComboHeight->setCurrentIndex(Line.section(",",1,1).toUInt());
      }
      else if(Setting == "Edit_MaxWidth")
        EditMaxWidth->setText(Line);
      else if(Setting == "Edit_MinWidth")
        EditMinWidth->setText(Line);
    }
    file.close();
    slotRealizationChanged(ComboRealize->currentIndex());
  }

  return result;
}

// -----------------------------------------------------------------------
// Saves the settings in the settings file.
bool QucsFilter::saveApplSettings(QDir *homeDir)
{
  QFile file(homeDir->absoluteFilePath("filterrc"));
  if(!file.open(QIODevice::WriteOnly))
    return false;

  QTextStream stream(&file);

  stream << "Settings file, QucsFilter " PACKAGE_VERSION "\n"
         << "FilterWindow=" << x() << ',' << y() << '\n'
         << "Combo_Realize=" << ComboRealize->currentIndex() << '\n'
         << "Combo_Type=" << ComboType->currentIndex() << '\n'
         << "Combo_Class=" << ComboClass->currentIndex() << '\n'
         << "Edit_Order=" << EditOrder->text() << '\n'
         << "Edit_Corner=" << EditCorner->text() << '\n'
         << "Combo_Corner=" << ComboCorner->currentIndex() << '\n'
         << "Edit_Stop=" << EditStop->text() << '\n'
         << "Combo_Stop=" << ComboStop->currentIndex() << '\n'
         << "Edit_BandStop=" << EditBandStop->text() << '\n'
         << "Combo_BandStop=" << ComboBandStop->currentIndex() << '\n'
         << "Edit_Ripple=" << EditRipple->text() << '\n'
         << "Edit_Impedance=" << EditImpedance->text() << '\n'
         << "Radio_Topology=" << (radioPi->isChecked() ? '0' : '1') << '\n'
         << "Check_Microstrip=" << (checkMicrostrip->isChecked() ? '1' : '0') << '\n'
         << "Combo_Er=" << ComboEr->lineEdit()->text() << '\n'
         << "Edit_Thickness=" << EditThickness->text() << ',' << ComboThickness->currentIndex() << '\n'
         << "Edit_Height=" << EditHeight->text() << ',' << ComboHeight->currentIndex() << '\n'
         << "Edit_MinWidth=" << EditMinWidth->text() << '\n'
         << "Edit_MaxWidth=" << EditMaxWidth->text() << '\n';

  file.close();
  return true;
}
