/***************************************************************************
  copyright: (C) 2010 by Michael Margraf
 ***************************************************************************/

#include <QString>
#include <QMessageBox>

#include "config.h"
#include "active_filter.h"

// --------------------------------------------------------------------
// active RC filter using multi-feedback topology with operational amplifiers
QByteArray Active_Filter::createSchematic(tFilter *Filter, double gain)
{
  int i, x;
  double a, b, C1, C2, R1, R2, R3, omega;

  // Even order Chebyshev filters have ripple above 0dB, let's compensate it.
  if(Filter->Type == TYPE_CHEBYSHEV)
    if((Filter->Order & 1) == 0)
      gain /= sqrt(Filter->Ripple);

  if(Filter->Class == CLASS_BANDPASS)
    return createBandpassSchematic(Filter, gain);

  // calculate gain per stage (one stage creates 2nd order)
  gain = pow(gain, 1.0/double(Filter->Order/2));

  omega = Filter->Frequency * M_TWO_PI;
  C1 = getE6value(2e-5 / Filter->Frequency);

  QString s = QUCSFILE_ID PACKAGE_VERSION ">\n";

  x = 60;
  s += "<Components>\n";
  s += QString("Vac V1 1 %1 270 18 -26 0 0 \"1 V\" 1\n").arg(x);
  s += QString("GND * 1 %1 300 0 0 0 0\n").arg(x);

  x += 60;
  for(i=1; i<=Filter->Order/2; i++) {
    a = getQuadraticNormValues(i, Filter, b);

    s += QString("OpAmp OP1 1 %1 200 -18 40 1 0 \"1e6\" 0\n").arg(x+240);
    s += QString("GND * 1 %1 220 0 0 0 0\n").arg(x+210);
    s += QString("GND * 1 %1 300 0 0 0 0\n").arg(x+70);

    switch(Filter->Class) {
      case CLASS_LOWPASS:
        C2 = getE6value(4.1 * b * (1.0+gain) * C1 / a / a);
        R2 = (a*C2 - sqrt(a*a*C2*C2 - 4.0*C1*C2*b*(1.0+gain))) / 2.0/C1/C2/omega;
        R1 = R2 / gain;
        R3 = b / (C1*C2*R2*omega*omega);

        s += QString("R R1 1 %1 180 -26 -59 1 0 \"%2\" 1\n").arg(x).arg(R1);
        s += QString("R R2 1 %1 130 15 -26 0 1 \"%2\" 1\n").arg(x+70).arg(R2);
        s += QString("R R3 1 %1 180 -26 15 0 0 \"%2\" 1\n").arg(x+130).arg(R3);
        s += QString("C C1 1 %1 130 17 -26 0 1 \"%2F\" 1\n").arg(x+190).arg(num2str(C1));
        s += QString("C C2 1 %1 270 17 -26 0 1 \"%2F\" 1\n").arg(x+70).arg(num2str(C2));
        break;
      case CLASS_HIGHPASS:
        C2 = C1 / gain;
        R1 = (2.0 + gain) / a / omega / C2;
        R2 = a / b / omega / C2 / (2.0 + gain);
        s += QString("R R1 1 %1 130 17 -26 0 1 \"%2\" 1\n").arg(x+190).arg(R1);
        s += QString("R R2 1 %1 270 17 -26 0 1 \"%2\" 1\n").arg(x+70).arg(R2);
        s += QString("C C1 1 %1 180 -26 -59 1 0 \"%2F\" 1\n").arg(x).arg(num2str(C1));
        s += QString("C C2 1 %1 130 15 -26 0 1 \"%2F\" 1\n").arg(x+70).arg(num2str(C2));
        s += QString("C C3 1 %1 180 -26 15 0 0 \"%2F\" 1\n").arg(x+130).arg(num2str(C2));
        break;
      default:
        QMessageBox::critical(0, "Error", "Filter class not supported.");
        return QByteArray();
    }
    x += 310;
  }

  if(Filter->Order & 1) {
    a = getQuadraticNormValues(0, Filter, b);

    s += QString("OpAmp OP1 1 %1 200 -23 -60 0 0 \"1e6\" 0\n").arg(x+120);
    s += QString("GND * 1 %1 300 0 0 0 0\n").arg(x+30);

    switch(Filter->Class) {
      case CLASS_LOWPASS:
        R1 = a / omega / C1;
        s += QString("R R1 1 %1 180 -26 -59 1 0 \"%2\" 1\n").arg(x).arg(R1);
        s += QString("C C1 1 %1 270 17 -26 0 1 \"%2F\" 1\n").arg(x+30).arg(num2str(C1));
        break;
      case CLASS_HIGHPASS:
        R1 = 1.0 / a / omega / C1;
        s += QString("R R1 1 %1 270 17 -26 0 1 \"%2\" 1\n").arg(x+30).arg(R1);
        s += QString("C C1 1 %1 180 -26 -59 1 0 \"%2F\" 1\n").arg(x).arg(num2str(C1));
        break;
      default:
        QMessageBox::critical(0, "Error", "Filter class not supported.");
        return QByteArray();
    }
    x += 190;
  }


  x -= 10;
  s += QString("R Rload 1 %1 270 15 -26 0 1 \"1k\" 1\n").arg(x);
  s += QString("GND * 1 %1 300 0 0 0 0\n").arg(x);
  s += QString(".AC AC1 1 80 350 0 41 0 0 \"log\" 1 \"%1Hz\" 1 \"%2Hz\" 1 \"501\" 1 \"no\" 0\n").arg(num2str(0.1*Filter->Frequency)).arg(num2str(10.0*Filter->Frequency));
  s += "</Components>\n";

  s += "<Wires>\n";

  // connect left source
  s += QString("60 180 60 240 \"\" 0 0 0\n");
  s += QString("60 180 90 180 \"\" 0 0 0\n");

  // connect right load resistor
  s += QString("%1 200 %2 240 \"output\" %3 180 30 \"\"\n").arg(x).arg(x).arg(x+30);
  s += QString("%1 200 %2 200 \"\" 0 0 0\n").arg(x-20).arg(x);

  // wires between components
  x = 150;
  for(i = 1; i < Filter->Order; i+=2) {
    s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x).arg(x+70);
    s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x+130).arg(x+180);
    s += QString("%1 160 %2 180 \"\" 0 0 0\n").arg(x+40).arg(x+40);
    s += QString("%1 180 %2 240 \"\" 0 0 0\n").arg(x+40).arg(x+40);
    s += QString("%1 160 %2 180 \"\" 0 0 0\n").arg(x+160).arg(x+160);
    s += QString("%1 100 %2 100 \"\" 0 0 0\n").arg(x+40).arg(x+250);
    s += QString("%1 100 %2 200 \"\" 0 0 0\n").arg(x+250).arg(x+250);
    x += 310;
  }

  if(Filter->Order & 1) {
    s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x).arg(x+60);
    s += QString("%1 180 %2 240 \"\" 0 0 0\n").arg(x).arg(x);
    s += QString("%1 220 %2 250 \"\" 0 0 0\n").arg(x+60).arg(x+60);
    s += QString("%1 250 %2 250 \"\" 0 0 0\n").arg(x+60).arg(x+130);
    s += QString("%1 200 %2 250 \"\" 0 0 0\n").arg(x+130).arg(x+130);
  }

  s += "</Wires>\n";

  s += "<Diagrams>\n";
  s += "</Diagrams>\n";

  s += "<Paintings>\n";

  s += QString("Text 270 350 12 #000000 0 \"active ");
  s += Filter::getDescription(Filter) + "\"\n";
  s += "</Paintings>\n";

  return s.toUtf8();
}

// --------------------------------------------------------------------
QByteArray Active_Filter::createBandpassSchematic(tFilter *Filter, double gain)
{
  int i, x;
  double a, b, Q, C1, C2, R1, R2, R3, omega, freq, beta, gamma, rho;

  // calculate gain per stage (number of stages equals order)
  gain = pow(gain, 1.0/double(Filter->Order));

  freq = sqrt(Filter->Frequency * Filter->Frequency2);
  omega = M_TWO_PI * freq;
  Q = freq / fabs(Filter->Frequency2 - Filter->Frequency);
  C1 = getE6value(2e-5 / freq);

  QString s(QUCSFILE_ID PACKAGE_VERSION ">\n");

  x = 60;
  s += "<Components>\n";
  s += QString("Vac V1 1 %1 270 18 -26 0 0 \"1 V\" 1\n").arg(x);
  s += QString("GND * 1 %1 300 0 0 0 0\n").arg(x);

  x += 60;
  for(i=1; i<=Filter->Order/2; i++) {
    a = getQuadraticNormValues(i, Filter, b);
    b = 1.0 / b;
    a *= b;

    beta = b + 4.0*Q*Q;
    beta = sqrt(0.5*(beta + sqrt(beta*beta - 4.0*a*a*Q*Q))) / a;
    gamma = a*beta / Q;
    gamma = 0.5*(gamma + sqrt(gamma*gamma-4.0));

    rho = gain*sqrt(b) / Q;
    beta = gamma / beta;
    gamma *= gamma;

    a = (rho*beta - gamma) * C1;
    C2 = a / gamma;
    if(C2 < 0.0)
      C2 = C1;
    R1 = 1.0 / (rho*omega*C1);
    R2 = (1.0/C1 + 1.0/C2) / (beta*omega);
    R3 = beta / ((gamma*C2 - a)*omega);

    s += QString("R R1 1 %1 180 -26 -59 1 0 \"%2\" 1\n").arg(x).arg(R1);
    s += QString("R R2 1 %1 130 17 -26 0 1 \"%2\" 1\n").arg(x+190).arg(R2);
    s += QString("R R3 1 %1 270 17 -26 0 1 \"%2\" 1\n").arg(x+70).arg(R3);
    s += QString("C C1 1 %1 130 15 -26 0 1 \"%2F\" 1\n").arg(x+70).arg(num2str(C1));
    s += QString("C C2 1 %1 180 -26 15 0 0 \"%2F\" 1\n").arg(x+130).arg(num2str(C2));
    s += QString("OpAmp OP1 1 %1 200 -18 40 1 0 \"1e6\" 0\n").arg(x+240);
    s += QString("GND * 1 %1 220 0 0 0 0\n").arg(x+210);
    s += QString("GND * 1 %1 300 0 0 0 0\n").arg(x+70);
    x += 310;

    gamma = 1.0/gamma;
    beta *= gamma;
    a = (rho*beta - gamma) * C1;
    C2 = a / gamma;
    if(C2 < 0.0)
      C2 = C1;
    R1 = 1.0 / (rho*omega*C1);
    R2 = (1.0/C1 + 1.0/C2) / (beta*omega);
    R3 = beta / ((gamma*C2 - a)*omega);

    s += QString("R R1 1 %1 180 -26 -59 1 0 \"%2\" 1\n").arg(x).arg(R1);
    s += QString("R R2 1 %1 130 17 -26 0 1 \"%2\" 1\n").arg(x+190).arg(R2);
    s += QString("R R3 1 %1 270 17 -26 0 1 \"%2\" 1\n").arg(x+70).arg(R3);
    s += QString("C C1 1 %1 130 15 -26 0 1 \"%2F\" 1\n").arg(x+70).arg(num2str(C1));
    s += QString("C C2 1 %1 180 -26 15 0 0 \"%2F\" 1\n").arg(x+130).arg(num2str(C2));
    s += QString("OpAmp OP1 1 %1 200 -18 40 1 0 \"1e6\" 0\n").arg(x+240);
    s += QString("GND * 1 %1 220 0 0 0 0\n").arg(x+210);
    s += QString("GND * 1 %1 300 0 0 0 0\n").arg(x+70);
    x += 310;
  }

  if(Filter->Order & 1) {
    a = getQuadraticNormValues(0, Filter, b);
    a = 1.0 / a;

    rho = gain*a / Q;
    beta = a / Q;
    gamma = 1.0;

    a = (rho*beta - gamma) * C1;
    C2 = a / gamma;
    if(C2 < 0.0)
      C2 = C1;
    R1 = 1.0 / (rho*omega*C1);
    R2 = (1.0/C1 + 1.0/C2) / (beta*omega);
    R3 = beta / ((gamma*C2 - a)*omega);

    s += QString("R R1 1 %1 180 -26 -59 1 0 \"%2\" 1\n").arg(x).arg(R1);
    s += QString("R R2 1 %1 130 17 -26 0 1 \"%2\" 1\n").arg(x+190).arg(R2);
    s += QString("R R3 1 %1 270 17 -26 0 1 \"%2\" 1\n").arg(x+70).arg(R3);
    s += QString("C C1 1 %1 130 15 -26 0 1 \"%2F\" 1\n").arg(x+70).arg(num2str(C1));
    s += QString("C C2 1 %1 180 -26 15 0 0 \"%2F\" 1\n").arg(x+130).arg(num2str(C2));
    s += QString("OpAmp OP1 1 %1 200 -18 40 1 0 \"1e6\" 0\n").arg(x+240);
    s += QString("GND * 1 %1 220 0 0 0 0\n").arg(x+210);
    s += QString("GND * 1 %1 300 0 0 0 0\n").arg(x+70);
    x += 310;
  }


  x -= 10;
  s += QString("R Rload 1 %1 270 15 -26 0 1 \"1k\" 1\n").arg(x);
  s += QString("GND * 1 %1 300 0 0 0 0\n").arg(x);
  s += QString(".AC AC1 1 80 350 0 41 0 0 \"log\" 1 \"%1Hz\" 1 \"%2Hz\" 1 \"501\" 1 \"no\" 0\n").arg(num2str(0.1*Filter->Frequency)).arg(num2str(10.0*Filter->Frequency));
  s += "</Components>\n";

  s += "<Wires>\n";

  // connect left source
  s += QString("60 180 60 240 \"\" 0 0 0\n");
  s += QString("60 180 90 180 \"\" 0 0 0\n");

  // connect right load resistor
  s += QString("%1 200 %2 240 \"output\" %3 180 30 \"\"\n").arg(x).arg(x).arg(x+30);
  s += QString("%1 200 %2 200 \"\" 0 0 0\n").arg(x-20).arg(x);

  // wires between components
  x = 150;
  for(i=0; i<Filter->Order; i++) {
    s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x).arg(x+70);
    s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x+130).arg(x+180);
    s += QString("%1 160 %2 180 \"\" 0 0 0\n").arg(x+40).arg(x+40);
    s += QString("%1 180 %2 240 \"\" 0 0 0\n").arg(x+40).arg(x+40);
    s += QString("%1 160 %2 180 \"\" 0 0 0\n").arg(x+160).arg(x+160);
    s += QString("%1 100 %2 100 \"\" 0 0 0\n").arg(x+40).arg(x+250);
    s += QString("%1 100 %2 200 \"\" 0 0 0\n").arg(x+250).arg(x+250);
    x += 310;
  }

  s += "</Wires>\n";

  s += "<Diagrams>\n";
  s += "</Diagrams>\n";

  s += "<Paintings>\n";

  s += QString("Text 270 350 12 #000000 0 \"active ");
  s += Filter::getDescription(Filter) + "\"\n";
  s += "</Paintings>\n";

  return s.toUtf8();
}
