/***************************************************************************
  copyright: (C) 2010 by Michael Margraf
 ***************************************************************************/

#include <QString>
#include <QMessageBox>

#include "config.h"
#include "cline_filter.h"

// lateral coupled transmission line bandpass filter
CoupledLine_Filter::CoupledLine_Filter()
{
}

// -------------------------------------------------------------------
QByteArray CoupledLine_Filter::createSchematic(tFilter *Filter, tSubstrate *Substrate)
{
  int i, x, y, dx;
  double *filterValues, Value, gap, len, dl, width, er_eff, Z0e, Z0o;
  double freq = (Filter->Frequency2 + Filter->Frequency) / 2.0;
  double Bandwidth = M_PI_2 * fabs(Filter->Frequency2 - Filter->Frequency) / freq;

  width = gap = er_eff = 1.0;

  filterValues = getNormValues(Filter);
  if(filterValues == 0)
    return QByteArray();  // empty array means errors

  if(Bandwidth > 0.4)
    QMessageBox::warning(0, QObject::tr("Warning"),
            QObject::tr("Result will be inaccurate, because bandwidth is too large."));

  QString s(QUCSFILE_ID PACKAGE_VERSION ">\n<Components>\n");

  x = 60;
  s += QString("Pac P1 1 %1 350 18 -26 0 0 \"1\" 1 \"%2\" 1\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 380 0 0 0 0\n").arg(x);

  x -= 30;
  y  = 170;
  len = dl = 0.0;
  for(i=0; i<=Filter->Order; i++) {
    x += 90;

    // characteristic admittance of J-inverter (normalized to Y0)
    if(i == 0)
      Value = sqrt(Bandwidth / filterValues[1]);
    else if(i == Filter->Order)
      Value = sqrt(Bandwidth / filterValues[i]);
    else
      Value = Bandwidth / sqrt(filterValues[i] * filterValues[i+1]);

    Z0e = (1.0 + Value + Value*Value) * Filter->Impedance;
    Z0o = (1.0 - Value + Value*Value) * Filter->Impedance;

    if(Substrate) {
      sythesizeCoupledMicrostrip(Z0e, Z0o, freq, Substrate, width, gap, er_eff);
      if((width < 1e-7) || (gap < 1e-7)) {
        QMessageBox::critical(0, QObject::tr("Error"), QObject::tr("Filter can't be created:\nwidth = %1, gap = %2").arg(width).arg(gap));
        return QByteArray();
      }
      dl = getMicrostripOpen(width / Substrate->height, Substrate->er, er_eff);
    }

    len = LIGHTSPEED / freq / 4.0 / sqrt(er_eff);

    if(Substrate) {
      y += 40;
      len -= dl * Substrate->height;
      s += QString("MCOUPLED MS1 1 %1 %2 -24 -115 0 0 \"Sub1\" 0 \"%3\" 1 \"%4\" 1 \"%5\" 1\n").arg(x).arg(y).arg(num2str(width, "m")).arg(num2str(len, "m")).arg(num2str(gap, "m"));
      s += QString("MOPEN MS1 1 %1 %2 -12 15 1 2 \"Sub1\" 0 \"%3\" 1\n").arg(x-60).arg(y+30).arg(num2str(width, "m"));
      s += QString("MOPEN MS1 1 %1 %2 -26 15 1 0 \"Sub1\" 0 \"%3\" 1\n").arg(x+60).arg(y-30).arg(num2str(width, "m"));
      y += 20;
      x += 60;
    }
    else {
      y += 20;
      s += QString("CTLIN Line1 1 %1 %2 -26 16 0 0 \"%3\" 1 \"%4\" 1 \"%5\" 1\n").arg(x).arg(y).arg(Z0e).arg(Z0o).arg(num2str(len, "m"));
    }
  }


  if(Substrate)
    x += 20;
  else
    x += 80;
  s += QString("Pac P2 1 %1 %2 18 -26 0 0 \"2\" 1 \"%3\" 1 \"0\" 0\n").arg(x).arg(y+40).arg(Filter->Impedance);
  s += QString("GND * 1 %1 %2 0 0 0 0\n").arg(x).arg(y+70);

  Value = 2.0 * Filter->Frequency  - Filter->Frequency2;
  if(Value < 1e-6)
    Value = 0.05*Filter->Frequency;
  s += QString(".SP SP1 1 50 430 0 67 0 0 \"lin\" 1 \"%2Hz\" 1 \"%3Hz\" 1 \"500\" 1 \"no\" 0 \"1\" 0 \"2\" 0\n").arg(num2str(Value)).arg(num2str(2.0 * Filter->Frequency2 - Filter->Frequency));
  if(Substrate)
    s += QString("SUBST Sub1 1 220 470 -30 24 0 0 \"%1\" 1 \"%2\" 1 \"%3\" 1 \"%4\" 1 \"%5\" 1 \"0\" 0\n").arg(Substrate->er).arg(num2str(Substrate->height, "m")).arg(num2str(Substrate->thickness, "m")).arg(Substrate->tand).arg(Substrate->resistivity);
  s += QString("Eqn Eqn1 1 310 507 0 8 0 0 \"S21_dB=dB(S[2,1])\" 1 \"S11_dB=dB(S[1,1])\" 1 \"S21_phase=wphase(S[2,1])\" 1 \"yes\" 0\n");
  s += "</Components>\n";

  s += "<Wires>\n";

  // connect left source
  s += QString("60 180 60 320 \"\" 0 0 0\n");
  s += QString("60 180 90 180 \"\" 0 0 0\n");

  // connect right source
  y += 10;
  s += QString("%1 %2 %3 %4 \"\" 0 0 0\n").arg(x-50).arg(y).arg(x).arg(y);

  // wires between components
  x = 150;
  y = 200;
  dx = 30;
  if(Substrate) {
    dx = 90;
    y += 40;
  }
  for(i = 0; i < Filter->Order; i++) {
    s += QString("%1 %2 %3 %4 \"\" 0 0 0\n").arg(x).arg(y).arg(x+dx).arg(y);
    if(Substrate) {
      x += 150;
      y += 60;
    }
    else {
      x += 90;
      y += 20;
    }
  }

  s += "</Wires>\n";

  s += "<Diagrams>\n</Diagrams>\n";

  s += "<Paintings>\n";
  s += QString("Text 310 420 12 #000000 0 \"");
  s += Filter::getDescription(Filter);
  s += QString(" \\n impedance %3 ohms\"\n").arg(Filter->Impedance);
  s += "</Paintings>\n";

  return s.toUtf8();
}
