/***************************************************************************
  copyright: (C) 2015 by Michael Margraf
 ***************************************************************************/

#include <QString>
#include <QMessageBox>

#include "config.h"
#include "coupledlc_filter.h"

// bandpass filter with C-coupled LC parallel resonance circuits
CoupledLC_Filter::CoupledLC_Filter()
{
}

// ====================================================================
// This is the main function. It creates an LC filter.
// Input parameters:
//       Frequency  - band start frequency
//       Frequency2 - band stop frequency
QByteArray CoupledLC_Filter::createSchematic(tFilter *Filter)
{
  int i, x, yc, yl;
  double *filterValues, Value, Value2, Omega, Bandwidth, L, f0, k;

  Omega = (Filter->Frequency2 + Filter->Frequency) * M_PI;
  Bandwidth = 0.5 * fabs(Filter->Frequency2 - Filter->Frequency) / Omega;

  filterValues = getNormValues(Filter);
  if(filterValues == 0)
    return QByteArray();  // empty array means errors

  if(Bandwidth > 0.035) {
    QMessageBox::critical(0, "Error", "Bandwidth too large for this filter.");
    return QByteArray();
  }

  L = getE6value(Filter->Impedance / Omega);

  Value  = 0.5/M_PI - Bandwidth;
  Value2 = 0.5/M_PI + Bandwidth;
  f0 = Value + Value2 - sqrt((Value2-Value)*(Value2-Value) + Value*Value2);
  k = (f0/Value - f0/Value2)*f0*M_TWO_PI * Filter->Impedance / (L * Omega);

  QString s(QUCSFILE_ID PACKAGE_VERSION ">\n<Components>\n");

  x = 20;
  yc = 320;
  yl = 240;
  s += QString("Pac P1 1 %1 320 18 -26 0 0 \"1\" 1 \"%2\" 1\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 350 0 0 0 0\n").arg(x);

  // first series capacitance
  Value  = sqrt(k/(filterValues[1] - k));
  s += QString("C C1 1 %1 %2 -27 -52 0 0 \"%3F\" 1\n").arg(x+30).arg(yl).arg(num2str(Value / (Filter->Impedance * Omega)));
  Value /= Value*Value + 1.0;

	x = 170;
  for(i=1; i<=Filter->Order; i++) {
    if(i == Filter->Order) {
      Value  = sqrt(k/(filterValues[1] - k));
      Value2 = Filter->Impedance / (L * Omega) - Value/(Value*Value + 1.0);
      Value2 -= k/sqrt(filterValues[1]*filterValues[2]);
    }
    else {
      Value2 = Filter->Impedance / (L * Omega) - Value;   // parallel capacitance
      Value  = k/sqrt(filterValues[i]*filterValues[i+1]); // series capacitance
      Value2 -= Value;
    }

    // parallel resonance circuit
    s += QString("L L1 1 %1 %2 8 -26 0 1 \"%3H\" 1\n").arg(x).arg(yc).arg(num2str(L));
    s += QString("C C1 1 %1 %2 -8 46 0 1 \"%3F\" 1\n").arg(x-30).arg(yc).arg(num2str(Value2 / (Filter->Impedance*Omega)));
    s += QString("GND * 1 %1 %2 0 0 0 0\n").arg(x).arg(yc + 30);

    // series capacitance
    s += QString("C C1 1 %1 %2 -27 -52 0 0 \"%3F\" 1\n").arg(x).arg(yl).arg(num2str(Value / (Filter->Impedance*Omega)));
    x += 120;
  }

  x -= 30;
  s += QString("Pac P2 1 %1 320 18 -26 0 0 \"2\" 1 \"%2\" 1 \"0\" 0\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 350 0 0 0 0\n").arg(x);

  yc += 100;
  Value = Filter->Frequency / 10.0;
  Value2 = 10.0 * Filter->Frequency2;
  s += QString(".SP SP1 1 40 %1 0 67 0 0 \"log\" 1 \"%2Hz\" 1 \"%3Hz\" 1 \"601\" 1 \"no\" 0 \"1\" 0 \"2\" 0\n").arg(yc).arg(num2str(Value)).arg(num2str(Value2));
  s += QString("Eqn Eqn1 1 200 %1 0 8 0 0 \"S21_dB=dB(S[2,1])\" 1 \"S11_dB=dB(S[1,1])\" 1 \"S21_phase=wphase(S[2,1])\" 1 \"yes\" 0\n").arg(yc+17);
  s += "</Components>\n";

  s += "<Wires>\n";

  // connect left source
  x = 20;
  s += QString("%1 240 %2 290 \"\" 0 0 0\n").arg(x).arg(x);

  // wires down to shunt components
  x = 140;
  for(i = 0; i < Filter->Order; i++) {
    s += QString("%1 240 %2 290 \"\" 0 0 0\n").arg(x).arg(x);
    x += 120;
  }

  // horizontal wires for series components
	x = 170;
  for(i = 0; i < Filter->Order; i++) {
    s += QString("%1 240 %2 240 \"\" 0 0 0\n").arg(x-90).arg(x-30);
    s += QString("%1 290 %2 290 \"\" 0 0 0\n").arg(x-30).arg(x);
    s += QString("%1 350 %2 350 \"\" 0 0 0\n").arg(x-30).arg(x);
    x += 120;
  }

  // connect right source
	x -= 30;
  s += QString("%1 240 %2 240 \"\" 0 0 0\n").arg(x-60).arg(x);
  s += QString("%1 240 %2 290 \"\" 0 0 0\n").arg(x).arg(x);

  s += "</Wires>\n";
  s += "<Diagrams>\n</Diagrams>\n";
  s += "<Paintings>\n";

  s += QString("Text 380 %1 12 #000000 0 \"").arg(yc);
  s += Filter::getDescription(Filter);
  s += QString(" \\n impedance matching %1 ohms\"\n").arg(Filter->Impedance);
  s += "</Paintings>\n";

  return s.toUtf8();
}
