/***************************************************************************
  copyright: (C) 2010 by Michael Margraf
 ***************************************************************************/

#include <QString>
#include <QMessageBox>

#include "config.h"
#include "eqn_filter.h"

Equation_Filter::Equation_Filter()
{
}

// ====================================================================
// This is the main function. It creates the filter schematic.
// Input parameters:
//       Class      - CLASS_LOWPASS
//                    CLASS_HIGHPASS
//                    CLASS_BANDPASS
//                    CLASS_BANDSTOP
//       Impedance  - input and output reference impedance in ohms
//       Order      - order of the filter
//       Frequency  - corner frequency (lowpass and highpass) or
//                    band start frequency (bandpass and bandstop)
//       Frequency2 - band stop frequency (only for bandpass and bandstop)
QByteArray Equation_Filter::createSchematic(tFilter *Filter)
{
  int i;
  double a, b, Omega, Bandwidth;
  if((Filter->Class == CLASS_BANDPASS) || (Filter->Class == CLASS_BANDSTOP))
    Omega = sqrt(Filter->Frequency2 * Filter->Frequency);
  else {
    Filter->Frequency2 = 0.0;
    Omega = Filter->Frequency;
  }
  Omega *= M_TWO_PI;   // angular frequency
  Bandwidth = M_TWO_PI * fabs(Filter->Frequency2 - Filter->Frequency);

  QString s(QUCSFILE_ID PACKAGE_VERSION ">\n<Components>\n");

  s += QString("Pac P1 1 100 30 18 -26 0 0 \"1\" 1 \"%1\" 1\n").arg(Filter->Impedance);
  s += QString("GND * 1 100 60 0 0 0 0\n");

  QString eqn = "2";  // because of 50ohms input and output

  // Even order Chebyshev filters have ripple above 0dB, let's compensate it.
  if(Filter->Type == TYPE_CHEBYSHEV)
    if((Filter->Order & 1) == 0)
      eqn = QString::number(2.0 / sqrt(Filter->Ripple));

  QString varS = "*S";
  switch(Filter->Class) {
    case CLASS_HIGHPASS:  // transform to highpass
      varS = "/S";
      Omega = 1.0 / Omega;
      break;
    case CLASS_BANDPASS:  // transform to bandpass
      varS = QString("*(S+%1/S)").arg(Omega*Omega);
      Omega = Bandwidth;
      break;
    case CLASS_BANDSTOP:  // transform to bandstop
      varS = QString("/(S+%1/S)").arg(Omega*Omega);
      Omega = 1.0 / Bandwidth;
      break;
  }


  if(Filter->Order & 1) {
    a = getQuadraticNormValues(0, Filter, b);
    eqn += QString(" / (1 + %1%2)").arg(a / Omega).arg(varS);
  }
  for(i = 1; i <= Filter->Order/2; i++) {
    a = getQuadraticNormValues(i, Filter, b);
    eqn += QString(" / (1 + %1%2 + %3%4%5)")
           .arg(a/Omega).arg(varS).arg(b/Omega/Omega).arg(varS).arg(varS);
  }


  s += QString("RFEDD X1 1 260 0 -28 24 0 0 \"G\" 0 \"2\" 0 \"open\" 0 \"%1\" 1 \"0\" 1 \"%2\" 1 \"%3\" 1\n").arg(1.0/Filter->Impedance).arg(eqn).arg(Filter->Impedance);

  s += QString("Pac P2 1 400 30 18 -26 0 0 \"2\" 1 \"%1\" 1 \"0\" 0\n").arg(Filter->Impedance);
  s += QString("GND * 1 400 60 0 0 0 0\n");

  if((Filter->Class == CLASS_BANDPASS) || (Filter->Class == CLASS_BANDSTOP))
    a = 10.0 * Filter->Frequency2;
  else
    a = 10.0 * Filter->Frequency;
  s += QString(".SP SP1 1 110 150 0 67 0 0 \"log\" 1 \"%1Hz\" 1 \"%2Hz\" 1 \"500\" 1 \"no\" 0 \"1\" 0 \"2\" 0\n").arg(num2str(0.1*Filter->Frequency)).arg(num2str(a));
  s += QString("Eqn Eqn1 1 280 167 0 8 0 0 \"S21_dB=dB(S[2,1])\" 1 \"S21_phase=wphase(S[2,1])\" 1 \"yes\" 0\n");
  s += "</Components>\n";

  s += "<Wires>\n";

  // connect left source
  s += QString("100 0 230 0 \"\" 0 0 0\n");

  // connect right source
  s += QString("290 0 400 0 \"\" 0 0 0\n");

  s += "</Wires>\n";

  s += "<Diagrams>\n";
  s += "</Diagrams>\n";

  s += "<Paintings>\n";

  s += QString("Text 380 140 12 #000000 0 \"");
  s += Filter::getDescription(Filter);
  s += QString(" \\n impedance matching %1 ohms\"\n").arg(Filter->Impedance);
  s += "</Paintings>\n";

  return s.toUtf8();
}
