/***************************************************************************
  copyright: (C) 2016 by Michael Margraf
 ***************************************************************************/

#include "mywidgets.h"

#include <QLineEdit>

typedef const char*   pChar;
typedef const char**  ppChar;


// ---------------------------------------------------------------------
// ----------         lists with material properties          ----------
// ---------------------------------------------------------------------

// Rules for the list:
// First string in list is the property name.
// Value and material name must be separated by two spaces!

// relative dielectric constant
pChar List_er[] = {"er",
   "9.8  alumina (Al2O3)", "3.78  fused quartz", "3.38  RO4003",
   "2.2  RT/duroid 5880", "10.2  RT/duroid 6010LM", "2.1  teflon (PTFE)",
   "4.0  PVC", "2.3  PE", "4.5  FR4", "6.6  beryllia (BeO)",
   "8.7  aluminum nitride", "11.9  silicon", "12.9  GaAs", "12.4  InP", 0};

// specific resistance in ohms*meters
pChar List_rho[] = {"rho",
   "1.72e-8  copper", "2.4e-8  gold", "1.62e-8  silver", "12.4e-8  tin",
   "10.5e-8  platinum", "10.8e-8  palladium", "2.62e-8  aluminum",
   "6.9e-8  nickel", "3.9e-8  brass 66/34", "9.71e-8  iron",
   "6.0e-8  zinc", 0};

// dielectric loss factor (tangent delta)
pChar List_tand[] = {"tand",
   "2e-4  aluminia (Al2O3) @ 10GHz", "1e-4  fused quartz @ 10GHz",
   "2e-3  RO4003 @ 10GHz", "9e-4  RT/duroid 5880 @ 10GHz",
   "3e-4  teflon (PTFE) @ 3GHz", "5e-2  PVC @ 1MHz", "2e-4  PE @ 1MHz",
   "2e-2  FR4 @ 1GHz", "3e-4  beryllia @ 10GHz",
   "1e-3  aluminum nitride @ 10GHz", "0.015  silicon @ 10GHz",
   "0.002  GaAs @ 10GHz", "3e-4  InP @ 10GHz", 0};

// relative permeability of conductors
pChar List_murC[] = {"murC",
   "0.999994  copper", "0.999966  gold", "1.000022  aluminum",
   "1.000265  platinum", "300  nickel", "5e3  iron", "100  C steel", 0};


ppChar List_Properties[] =
  {(ppChar)&List_er, (ppChar)&List_tand, (ppChar)&List_rho, 0};


// ---------------------------------------------------------------------
// ----------       input line with pre-defined values        ----------
// ---------------------------------------------------------------------

QuantityBox::QuantityBox(ppChar list, QWidget *parent)
           : QComboBox(parent)
{
  setEditable(true);
  setCompleter(0);
  dblVal.setBottom(0.0);
  lineEdit()->setValidator(&dblVal);

  list++;  // first entry is name of quantity
  while(*list)
    addItem(*(list++));
  slotSetIndex();
  connect(this, SIGNAL(activated(const QString&)), SLOT(slotSetIndex()));
}

// ---------------------------------------------------------------------
void QuantityBox::slotSetIndex()
{
  setEditText(currentText().section(" ", 0, 0));
}


// ---------------------------------------------------------------------
// ----------           ComboBox for unit choice              ----------
// ---------------------------------------------------------------------

UnitComboBox::UnitComboBox(int quantity_, int index_)
{
  quantity = quantity_;

  switch(quantity) {
    case QUANTITY_LENGTH:
      addItem("km");
      addItem("m");
      addItem("cm");
      addItem("mm");
      addItem("m");
      addItem("mile");
      addItem("yard");
      addItem("foot");
      addItem("inch");
      addItem("mil");
      break;
    case QUANTITY_FREQUENCY:
      addItem("Hz");
      addItem("kHz");
      addItem("MHz");
      addItem("GHz");
      addItem("THz");
      break;
  }

  setCurrentIndex(index_);
}

// ------------------------------------------------------------------
QString UnitComboBox::string(QLineEdit *edit)
{
  QString s = edit->text();

  if(quantity == QUANTITY_LENGTH)
    switch(currentIndex()) {
      case 1:  // meters
        break;
      case 5:
        s = QString::number(edit->text().toDouble()*1.609e3); // mile
        break;
      case 6:
        s = QString::number(edit->text().toDouble()*0.9144);  // yard
        break;
      case 7:
        s = QString::number(edit->text().toDouble()*0.3048);  // foot
        break;
      case 8:
        s = QString::number(edit->text().toDouble()*2.54e-2); // inch
        break;
      default: s += currentText();
    }
  else
    s += currentText();

  return s;
}

// ------------------------------------------------------------------
// Returns value in meters or Hertz, respectively.
double UnitComboBox::value(QLineEdit *edit)
{
  double d = edit->text().toDouble();
  if(quantity == QUANTITY_LENGTH)
    switch(currentIndex()) {
      case 0:  return d*1e3;      // km
      case 2:  return d*1e-2;     // cm
      case 3:  return d*1e-3;     // mm
      case 4:  return d*1e-6;     // m
      case 5:  return d*1.609e3;  // mile
      case 6:  return d*0.9144;   // yard
      case 7:  return d*0.3048;   // foot
      case 8:  return d*2.54e-2;  // inch
      case 9:  return d*2.54e-5;  // mil
    }
  else
    switch(currentIndex()) {
      case 1:  return d*1e3;  // kHz
      case 2:  return d*1e6;  // MHz
      case 3:  return d*1e9;  // GHz
      case 4:  return d*1e12; // THz
    }

  return d; // without factor (meters, hertz, etc.)
}

// ------------------------------------------------------------------
// Sets value with respect to the unit.
// "value" is in meters or Hertz, respectively.
void UnitComboBox::setValue(double value, QLineEdit *edit)
{
  if(quantity == QUANTITY_LENGTH)
    switch(currentIndex()) {
      case 0:  value /= 1e3;  break;    // km
      case 2:  value /= 1e-2; break;    // cm
      case 3:  value /= 1e-3; break;    // mm
      case 4:  value /= 1e-6; break;    // m
      case 5:  value /= 1.609e3; break; // mile
      case 6:  value /= 0.9144; break;  // yard
      case 7:  value /= 0.3048; break;  // foot
      case 8:  value /= 2.54e-2; break; // inch
      case 9:  value /= 2.54e-5; break; // mil
    }
  else
    switch(currentIndex()) {
      case 1:  value /= 1e3; break;  // kHz
      case 2:  value /= 1e6; break;  // MHz
      case 3:  value /= 1e9; break;  // GHz
      case 4:  value /= 1e12; break; // THz
    }

  edit->setText(QString::number(value));
}

// ------------------------------------------------------------------
// Returns whether current unit is metric or British.
bool UnitComboBox::isMetric()
{
  return currentIndex() < 5;
}
