/***************************************************************************
  copyright: (C) 2010 by Michael Margraf
 ***************************************************************************/

#include <QString>
#include <QMessageBox>

#include "config.h"
#include "stepz_filter.h"

// -----------------------------------------------------------------------
// stepped-impedance transmission line lowpass filter
QByteArray StepImpedance_Filter::createSchematic(tFilter *Filter, tSubstrate *Substrate)
{
  int i, x;
  double Lpar, Cser;
  double *filterValues, len, width, er_eff_min, er_eff_max, Z0;
  double Omega = M_TWO_PI * Filter->Frequency;
  double Zlow  = Filter->Zlow;
  double Zhigh = Filter->Zhigh;

  er_eff_min = er_eff_max = width = 1.0;

  if(Substrate) {
    calcMicrostrip(Substrate, Substrate->minWidth, Filter->Frequency, er_eff_min, Zhigh);
    calcMicrostrip(Substrate, Substrate->maxWidth, Filter->Frequency, er_eff_max, Zlow);

    if((Substrate->er > 4.0) || (Substrate->height > 0.6))
      QMessageBox::warning(0, QObject::tr("Warning"),
          QObject::tr("High-impedance is %1 ohms, low-impedance is %2 ohms.\n"
                      "To get acceptable results it is recommended to use\n"
                      "a substrate with lower permittivity and larger height.\n").arg(Zhigh).arg(Zlow));

    getMicrostripStep(Substrate->maxWidth, Substrate->minWidth, Filter->Frequency,
                      Substrate, Lpar, Cser);
  }

  filterValues = getNormValues(Filter);
  if(filterValues == 0)
    return QByteArray();  // empty array means errors

  QString s(QUCSFILE_ID PACKAGE_VERSION ">\n<Components>\n");

  x = 0;
  s += QString("Pac P1 1 %1 310 18 -26 0 0 \"1\" 1 \"%2\" 1\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 340 0 0 0 0\n").arg(x);

  x += 60;
  for(i=1; i<=Filter->Order; i++) {
    // De-normalize values and calculate transmission line length.
    // The asin() is omitted here as it brings disadvantages only.
    // The same holds for potential improvements with tan().
    if((i & 1) == Filter->piType) {
      // replace a capacitor
      Z0 = Zlow;
      len = filterValues[i] / Filter->Impedance;
      if(Substrate) {
        if((i > 1) && (i < Filter->Order))
          len -= 2.0*Cser; // remove impedance step from both sides
        else
          len -= Cser;     // remove impedance step from one side only
        width = Substrate->maxWidth;
        len /= sqrt(er_eff_max);
      }
      len *= Zlow;
    }
    else {
      // replace an inductor
      Z0 = Zhigh;
      len = filterValues[i] * Filter->Impedance;
      if(Substrate) {
        if((i > 1) && (i < Filter->Order))
          len -= 2.0*Lpar; // remove impedance step from both sides
        else
          len -= Lpar;     // remove impedance step from one side only
        width = Substrate->minWidth;
        len /= sqrt(er_eff_min);
      }
      len /= Zhigh;
    }
    len *= LIGHTSPEED / Omega;

    if(Substrate) {
      if(i > 1) {
        s += QString("MSTEP MS4 1 %1 180 -26 -73 %2 \"Sub1\" 0 \"%3\" 1 \"%4\" 1\n").arg(x).arg(((i&1) == Filter->piType)?"1 2":"0 0").arg(num2str(Substrate->maxWidth, "m")).arg(num2str(Substrate->minWidth, "m"));
        x += 60;
      }
      s += QString("MLIN MS1 1 %1 180 -26 15 0 0 \"Sub1\" 0 \"%2\" 1 \"%3\" 1\n").arg(x).arg(num2str(width, "m")).arg(num2str(len, "m"));
      x += 60;
    }
    else {
      s += QString("TLIN Line1 1 %1 180 -26 20 0 0 \"%2\" 1 \"%3\" 1\n").arg(x).arg(Z0).arg(num2str(len, "m"));
      x += 90;
    }
  }


  if(Substrate)
    x += 30;
  s += QString("Pac P2 1 %1 310 18 -26 0 0 \"2\" 1 \"%2\" 1 \"0\" 0\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 340 0 0 0 0\n").arg(x);

  s += QString(".SP SP1 1 10 390 0 67 0 0 \"lin\" 1 \"%2Hz\" 1 \"%3Hz\" 1 \"496\" 1 \"no\" 0 \"1\" 0 \"2\" 0\n").arg(num2str(0.05*Filter->Frequency)).arg(num2str(5.0*Filter->Frequency));
  if(Substrate)
    s += QString("SUBST Sub1 1 190 430 -30 24 0 0 \"%1\" 1 \"%2\" 1 \"%3\" 1 \"%4\" 1 \"%5\" 1 \"0\" 0\n").arg(Substrate->er).arg(num2str(Substrate->height, "m")).arg(num2str(Substrate->thickness, "m")).arg(Substrate->tand).arg(Substrate->resistivity);
  s += QString("Eqn Eqn1 1 290 457 0 8 0 0 \"S21_dB=dB(S[2,1])\" 1 \"S11_dB=dB(S[1,1])\" 1 \"S21_phase=wphase(S[2,1])\" 1 \"yes\" 0\n");
  s += "</Components>\n";

  s += "<Wires>\n";

  // connect left source
  s += QString("0 180 0 280 \"\" 0 0 0\n");
  s += QString("0 180 30 180 \"\" 0 0 0\n");

  // connect right source
  s += QString("%1 180 %2 280 \"\" 0 0 0\n").arg(x).arg(x);
  s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x-60).arg(x);

  // wires between components
  if(Substrate == 0) {
    x = 90;
    for(i=1; i<Filter->Order; i++) {
      s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x).arg(x+30);
      x += 90;
    }
  }

  s += "</Wires>\n";

  s += "<Diagrams>\n</Diagrams>\n";

  s += "<Paintings>\n";
  s += QString("Text 290 380 12 #000000 0 \"");
  s += Filter::getDescription(Filter);
  s += QString(" \\n impedance matching %1 ohms\"\n").arg(Filter->Impedance);
  s += "</Paintings>\n";

  return s.toUtf8();
}
