/***************************************************************************
  copyright: (C) 2010 by Michael Margraf
 ***************************************************************************/

#include "config.h"
#include "stub_filter.h"

#include <qstring.h>
#include <qmessagebox.h>

// -----------------------------------------------------------------------
// quarter-wave stub transmission line filter
QByteArray Stub_Filter::createSchematic(tFilter *Filter, tSubstrate *Substrate)
{
  int i, x;
  double *g, zl, zl_stub, len, dl, width, er_eff, freq, tanTheta, h, J, Y, N2;

  if((Filter->Class == CLASS_LOWPASS) || (Filter->Class == CLASS_HIGHPASS))
    return createLowpassSchematic(Filter, Substrate);

  g = getNormValues(Filter);
  if(g == 0)
    return QByteArray();  // empty array means errors

  width = er_eff = 1.0;
  freq = 0.5*(Filter->Frequency2 + Filter->Frequency);
  len = 0.25*LIGHTSPEED / freq;  // quarter wavelength

  QString s(QUCSFILE_ID PACKAGE_VERSION ">\n<Components>\n");

  x = 0;
  s += QString("Pac P1 1 %1 310 18 -26 0 0 \"1\" 1 \"%2\" 1\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 340 0 0 0 0\n").arg(x);

  h = 2.0;  // can be used to vary the line impedances
  tanTheta = fabs(Filter->Frequency2 - Filter->Frequency) / freq;
  tanTheta = tan(M_PI_2*(1.0 - 0.5*tanTheta));
  N2 = 0.5*h*g[1]*tanTheta;
  N2 *= N2;

  x += 40;
  for(i=1; i<=Filter->Order; i++) {

    if(Filter->Class == CLASS_BANDPASS) {
      // taken from page 153 of:
      // JIA-SHENG HONG, M. J. LANCASTER,
      // "Microstrip Filters for RF/Microwave Applications",
      // John Wiley & Sons, Inc., 2001
      J = h*g[1];
      if(i == 1) {
        J = sqrt(J/g[2]);
        Y = (1.0-0.5*h)*g[1]*tanTheta;
      }
      else if(i < Filter->Order) {
        if(i == 2)
          J = sqrt(J/g[2]);
        else
          J /= sqrt(g[i-1]*g[i]);
        Y = sqrt(J*J + N2) - J;
        J = h*g[1];
        if(i < (Filter->Order-1))
          J /= sqrt(g[i]*g[i+1]);
        else
          J = sqrt(J/g[i]);
      }
      else {
        J = sqrt(J/g[i-1]);
        Y = (g[i] - 0.5*h*g[1])*tanTheta;
      }
      Y += sqrt(J*J + N2) - J;
      zl = Filter->Impedance / J;
      zl_stub = Filter->Impedance / Y;
    }
    else {  // bandstop
      zl = Filter->Impedance;
      tanTheta = fabs(Filter->Frequency2 - Filter->Frequency) / freq;
      zl_stub = Filter->Impedance / (M_PI_2*tanTheta*g[i]);
    }

    if(Substrate) {
      dl = 0.0;
      getMicrostrip(zl_stub, freq, Substrate, width, er_eff);
      if(Filter->Class == CLASS_BANDPASS)
        s += QString("GND * 1 %1 80 0 0 1 0\n").arg(x);
      else {  // bandstop
        dl = getMicrostripOpen(width/Substrate->height, Substrate->er, er_eff);
        dl *= Substrate->height;
        s += QString("MOPEN MS1 1 %1 50 10 -10 0 1 \"Sub1\" 0 \"%2\" 1\n")
             .arg(x).arg(num2str(width, "m"));
      }
      s += QString("MLIN MS1 1 %1 110 10 -25 0 1 \"Sub1\" 0 \"%2\" 1 \"%3\" 1\n")
           .arg(x).arg(num2str(width, "m")).arg(num2str(len/sqrt(er_eff)-dl, "m"));
      getMicrostrip(zl, freq, Substrate, width, er_eff);
      if(i < Filter->Order)
        s += QString("MLIN MS1 1 %1 180 -26 15 0 0 \"Sub1\" 0 \"%2\" 1 \"%3\" 1\n")
             .arg(x+30).arg(num2str(width, "m")).arg(num2str(len/sqrt(er_eff), "m"));
    }
    else {
      s += QString("TLIN Line1 1 %1 110 10 -25 1 1 \"%2\" 1 \"%3\" 1\n").arg(x).arg(zl_stub).arg(num2str(len, "m"));
      if(Filter->Class == CLASS_BANDPASS)
        s += QString("GND * 1 %1 80 0 0 1 0\n").arg(x);
      if(i < Filter->Order)
        s += QString("TLIN Line1 1 %1 180 -26 20 0 0 \"%2\" 1 \"%3\" 1\n").arg(x+30).arg(zl).arg(num2str(len, "m"));
    }
    x += 100;
  }


  x -= 60;
  s += QString("Pac P2 1 %1 310 18 -26 0 0 \"2\" 1 \"%2\" 1 \"0\" 0\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 340 0 0 0 0\n").arg(x);

  s += QString(".SP SP1 1 10 390 0 67 0 0 \"lin\" 1 \"%2Hz\" 1 \"%3Hz\" 1 \"496\" 1 \"no\" 0 \"1\" 0 \"2\" 0\n").arg(num2str(0.05*Filter->Frequency)).arg(num2str(5.0*Filter->Frequency));
  if(Substrate)
    s += QString("SUBST Sub1 1 190 430 -30 24 0 0 \"%1\" 1 \"%2\" 1 \"%3\" 1 \"%4\" 1 \"%5\" 1 \"0\" 0\n").arg(Substrate->er).arg(num2str(Substrate->height, "m")).arg(num2str(Substrate->thickness, "m")).arg(Substrate->tand).arg(Substrate->resistivity);
  s += QString("Eqn Eqn1 1 290 457 0 8 0 0 \"S21_dB=dB(S[2,1])\" 1 \"S11_dB=dB(S[1,1])\" 1 \"S21_phase=wphase(S[2,1])\" 1 \"yes\" 0\n");
  s += "</Components>\n";

  s += "<Wires>\n";

  // connect left source
  s += QString("0 180 0 280 \"\" 0 0 0\n");

  // connect right source
  s += QString("%1 180 %2 280 \"\" 0 0 0\n").arg(x).arg(x);
  s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x-40).arg(x);

  // wires between components
  x = 40;
  for(i=0; i<Filter->Order; i++) {
    s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x-40).arg(x);
    s += QString("%1 140 %2 180 \"\" 0 0 0\n").arg(x).arg(x);
    x += 100;
  }

  s += "</Wires>\n";

  s += "<Diagrams>\n</Diagrams>\n";

  s += "<Paintings>\n";
  s += QString("Text 290 380 12 #000000 0 \"");
  s += Filter::getDescription(Filter);
  s += QString("impedance matching %3 ohms\"\n").arg(Filter->Impedance);
  s += "</Paintings>\n";

  return s.toUtf8();
}

// -----------------------------------------------------------------------
QByteArray Stub_Filter::createLowpassSchematic(tFilter *Filter, tSubstrate *Substrate)
{
  int i, x;
  double *g, len, dl, freq, er_eff_max, er_eff_min;
  double Zlow  = Filter->Zlow;
  double Zhigh = Filter->Zhigh;

  g = getNormValues(Filter);
  if(g == 0)
    return QByteArray();  // empty array means errors

  freq = M_TWO_PI * Filter->Frequency;

  dl = 0.0;
  er_eff_max = er_eff_min = 1.0;
  len = 0.25*LIGHTSPEED / freq;  // quarter wavelength
  if(Substrate) {
    calcMicrostrip(Substrate, Substrate->minWidth, Filter->Frequency, er_eff_min, Zhigh);
    calcMicrostrip(Substrate, Substrate->maxWidth, Filter->Frequency, er_eff_max, Zlow);
    dl = getMicrostripOpen(Substrate->maxWidth/Substrate->height, Substrate->er, er_eff_max);
    dl *= Substrate->height;
  }

  QString s(QUCSFILE_ID PACKAGE_VERSION ">\n<Components>\n");

  x = 0;
  s += QString("Pac P1 1 %1 310 18 -26 0 0 \"1\" 1 \"%2\" 1 \"0 dBm\" 0\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 340 0 0 0 0\n").arg(x);

  x += 30;
  for(i=1; i<=Filter->Order; i++) {
    // De-normalize values and calculate transmission line length.
    if((i & 1) == Filter->piType) {
      // replace a capacitor
      len = Filter->Impedance / g[i];
      if(Filter->Class == CLASS_HIGHPASS) {
        s += QString("GND * 1 %1 80 0 0 1 0\n").arg(x+30);
        len = LIGHTSPEED / freq * atan(0.5*len/Zhigh);
        if(Substrate) {
          len /= sqrt(er_eff_min);
          s += QString("MLIN MS1 1 %1 110 12 -25 0 1 \"Sub1\" 0 \"%2\" 1 \"%3\" 1\n").arg(x+30).arg(num2str(Substrate->minWidth, "m")).arg(num2str(len, "m"));
        }
        else
        s += QString("TLIN Line1 1 %1 110 10 -25 1 1 \"%2\" 1 \"%3\" 1\n").arg(x+30).arg(Zhigh).arg(num2str(len, "m"));
      }
      else {
        len = LIGHTSPEED / freq * atan(Zlow/len);
        if(Substrate) {
          s += QString("MOPEN MS1 1 %1 50 12 -10 0 1 \"Sub1\" 0 \"%2\" 1\n").arg(x+30).arg(num2str(Substrate->maxWidth, "m"));
          len = len / sqrt(er_eff_max) - dl;
          s += QString("MLIN MS1 1 %1 110 12 -25 0 1 \"Sub1\" 0 \"%2\" 1 \"%3\" 1\n").arg(x+30).arg(num2str(Substrate->maxWidth, "m")).arg(num2str(len, "m"));
        }
        else
        s += QString("TLIN Line1 1 %1 110 10 -25 1 1 \"%2\" 1 \"%3\" 1\n").arg(x+30).arg(Zlow).arg(num2str(len, "m"));
      }
    }
    else {
      // replace an inductor
      len = Filter->Impedance * g[i];
      if(Filter->Class == CLASS_HIGHPASS) {
        len = Zlow/len;
        if(len > 1.0)
          len = 1.0;
        len = LIGHTSPEED / freq * asin(len);
        if(Substrate) {
          len /= sqrt(er_eff_max);
          s += QString("MLIN MS1 1 %1 180 -26 15 0 0 \"Sub1\" 0 \"%2\" 1 \"%3\" 1\n").arg(x).arg(num2str(Substrate->maxWidth, "m")).arg(num2str(len, "m"));
        }
        else
        s += QString("TLIN Line1 1 %1 180 -26 20 0 0 \"%2\" 1 \"%3\" 1\n").arg(x).arg(Zlow).arg(num2str(len, "m"));
      }
      else {
        len = len/Zhigh;
        if(len > 1.0)
          len = 1.0;
        len = LIGHTSPEED / freq * asin(len);
        if(Substrate) {
          len /= sqrt(er_eff_min);
          s += QString("MLIN MS1 1 %1 180 -26 15 0 0 \"Sub1\" 0 \"%2\" 1 \"%3\" 1\n").arg(x).arg(num2str(Substrate->minWidth, "m")).arg(num2str(len, "m"));
        }
        else
        s += QString("TLIN Line1 1 %1 180 -26 20 0 0 \"%2\" 1 \"%3\" 1\n").arg(x).arg(Zhigh).arg(num2str(len, "m"));
      }
    }

    x += 60;
  }


  x += 10;
  s += QString("Pac P2 1 %1 310 18 -26 0 0 \"2\" 1 \"%2\" 1 \"0\" 0\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 340 0 0 0 0\n").arg(x);

  s += QString(".SP SP1 1 10 390 0 67 0 0 \"lin\" 1 \"%2Hz\" 1 \"%3Hz\" 1 \"496\" 1 \"no\" 0 \"1\" 0 \"2\" 0\n").arg(num2str(0.05*Filter->Frequency)).arg(num2str(5.0*Filter->Frequency));
  if(Substrate) {
    s += QString("SUBST Sub1 1 190 430 -30 24 0 0 \"%1\" 1 \"%2\" 1 \"%3\" 1 \"%4\" 1 \"%5\" 1 \"0\" 0\n").arg(Substrate->er).arg(num2str(Substrate->height, "m")).arg(num2str(Substrate->thickness, "m")).arg(Substrate->tand).arg(Substrate->resistivity);
  s += QString("Eqn Eqn1 1 290 457 0 8 0 0 \"S21_dB=dB(S[2,1])\" 1 \"S11_dB=dB(S[1,1])\" 1 \"S21_phase=wphase(S[2,1])\" 1 \"yes\" 0\n");
  }
  else
    s += QString("Eqn Eqn1 1 170 407 0 8 0 0 \"S21_dB=dB(S[2,1])\" 1 \"S11_dB=dB(S[1,1])\" 1 \"S21_phase=wphase(S[2,1])\" 1 \"yes\" 0\n");
  s += "</Components>\n";

  s += "<Wires>\n";

  // connect left source
  s += QString("0 180 0 280 \"\" 0 0 0\n");

  // connect right source
  s += QString("%1 180 %2 280 \"\" 0 0 0\n").arg(x).arg(x);
  s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x-40).arg(x);

  // wires between components
  if(Filter->piType) {
    x = 60;
    i = 0;
  }
  else {
    x = 120;
    i = 2;
  }
  for(; i<Filter->Order; i+=2) {
    s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x-60).arg(x);
    s += QString("%1 140 %2 180 \"\" 0 0 0\n").arg(x).arg(x);
    x += 120;
  }

  s += "</Wires>\n";

  s += "<Diagrams>\n</Diagrams>\n";

  s += "<Paintings>\n";
  s += QString("Text 290 380 12 #000000 0 \"");
  s += Filter::getDescription(Filter);
  s += QString(" \\n impedance matching %1 ohms\"\n").arg(Filter->Impedance);
  s += "</Paintings>\n";

  return s.toUtf8();
}
