/***************************************************************************
  copyright: (C) 2016 by Michael Margraf
 ***************************************************************************/

#include <QString>
#include <QMessageBox>

#include "config.h"
#include "lc_diplexer.h"
#include "qf_poly.h"
#include "qf_comp.h"
#include "qf_filter.h"
#include "qf_legendre.h"

// --------------------------------------------------------------------
// Coefficients of Bessel diplexer taken from table C.4 (page 340) of
// Rabin Raut and M. N. S. Swamy,
// "Modern Analog Filter Analysis and Design",
// WILEY-VCH Verlag & Co. KGaA, 2010
double BesselDiplexer[5][6] = {
  /* 2 */  {1.5774, 0.4226, 0.0,    0.0,    0.0,    0.0},
  /* 3 */  {1.2550, 0.5528, 0.1922, 0.0,    0.0,    0.0},
  /* 4 */  {1.0598, 0.5116, 0.3181, 0.1104, 0.0,    0.0},
  /* 5 */  {0.9303, 0.4577, 0.3312, 0.2090, 0.0718, 0.0},
  /* 6 */  {0.8377, 0.4116, 0.3158, 0.2364, 0.1480, 0.0505}};

// --------------------------------------------------------------------
// This is the main function. It creates an LC diplexer.
// Input parameters:
//       Impedance  - input and output reference impedance in ohms
//       Order      - order of the filter
//       Frequency  - crossover frequency
QByteArray LC_Diplexer::createSchematic(tFilter *Filter, bool isTriplexer)
{
  int i, x, y;
  double Value, Value2, Omega, Bandwidth, relBandwidth, aVal[64], cVal[64], filterValues[64];

  if(Filter->Type == TYPE_LEGENDRE) {
    qf_spec spec;
    spec.ord = Filter->Order;
    qf_lgndr *filter = new qf_lgndr(&spec);
    filter->norm_values(filterValues);
    delete filter;
  }
  else if(Filter->Type == TYPE_BESSEL) {
    if(Filter->Order > 6) {
      QMessageBox::critical(0, QObject::tr("Error"), QObject::tr("Filter order must not be greater than 6."));
      return QByteArray();  // empty array means errors
    }
    for(i=0; i<Filter->Order; i++)
      filterValues[i] = BesselDiplexer[Filter->Order-2][i];
  }
  else {
    // calculate normalized Butterworth or Chebyshev coefficients, taken from:
    //   Matthaei, Young and Jones
    //   "Design of Microwave Filters, Impedance-Matching Networks and Coupling Structures"
    //   Volume 1, January 1963
    //   Stanford Research Institute, Menlo Park, California
    //   page 104ff
    // It uses the "singly terminated filter" approach.
    aVal[0] = 1.0;
    for(i=1; i<=Filter->Order; i++) {
      aVal[i-1] = sin(M_PI*double(2*i-1)/double(2*Filter->Order));
      Value = cos(M_PI*double(i)/double(2*Filter->Order));
      cVal[i-1] = Value * Value;
    }
    filterValues[Filter->Order-1] = aVal[0];

    if(Filter->Type == TYPE_CHEBYSHEV) {
      Value2 = sqrt(Filter->Ripple - 1.0);                      // = epsilon
      Value2 = sinh(asinh(1.0 / Value2) / double(Filter->Order)); // = gamma
      for(i=1; i<=Filter->Order; i++) {
        Value = sin(M_PI*double(i)/double(2*Filter->Order));
        cVal[i-1] *= Value2*Value2 + Value*Value;
      }
      filterValues[Filter->Order-1] /= Value2;
    }
    else if(Filter->Type != TYPE_BUTTERWORTH) {
      QMessageBox::critical(0, QObject::tr("Error"), QObject::tr("Diplexers are not supported for this filter type."));
      return QByteArray();  // empty array means errors
    }

    for(i=1; i<Filter->Order; i++)
      filterValues[Filter->Order-i-1]
        = aVal[i]*aVal[i-1] / cVal[i-1] / filterValues[Filter->Order-i];

    if(Filter->Type == TYPE_CHEBYSHEV) {
      // normaize filter to -3dB bandwidth
      Value = 1.0 / sqrt(Filter->Ripple - 1.0);
      if((Filter->Order & 1) == 0)
        Value *= sqrt(2.0*Filter->Ripple - 1.0);
      Value = cosh(acosh(Value) / double(Filter->Order));
      for(i=0; i<Filter->Order; i++)
        filterValues[i] *= Value;
    }
  }


  QString s(QUCSFILE_ID PACKAGE_VERSION ">\n<Components>\n");

  x = 60;
  s += QString("Pac P1 1 %1 320 18 -26 0 0 \"1\" 1 \"%2\" 1\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 350 0 0 0 0\n").arg(x);

  if(isTriplexer)
    relBandwidth = Filter->Frequency / Filter->Frequency2;
  else
    relBandwidth = 1.0;
  Bandwidth = M_TWO_PI*fabs(Filter->Frequency2 - Filter->Frequency);
  Omega = M_TWO_PI*Filter->Frequency;  // angular frequency
  for(i=0; i<Filter->Order; i++) {
    x = ((i+1) * 70) + 100;

    // de-normalize
    Value = filterValues[i];
    if(i & 1)
      Value /= Filter->Impedance;
    else
      Value *= Filter->Impedance;

    // components for low-pass
    Value2 = Value / Omega;
    if(i & 1) {
      s += QString("C C1 1 %1 320 17 -26 0 1 \"%2F\" 1\n").arg(x).arg(num2str(Value2));
      s += QString("GND * 1 %1 350 0 0 0 0\n").arg(x);
    }
    else
      s += QString("L L1 1 %1 240 -26 10 0 0 \"%2H\" 1\n").arg(x).arg(num2str(Value2));

    // components for high-pass
    Value2 = relBandwidth / Omega / Value;  // transform to highpass
    if(i & 1) {
      s += QString("L L1 1 %1 490 17 -26 0 1 \"%2H\" 1\n").arg(x).arg(num2str(Value2));
      s += QString("GND * 1 %1 520 0 0 0 0\n").arg(x);
    }
    else
      s += QString("C C1 1 %1 410 -27 10 0 0 \"%2F\" 1\n").arg(x).arg(num2str(Value2));

    if(isTriplexer) {
      // components for band-pass
      Value /= Bandwidth;    // transform to bandpass
      Value2 = 0.25 / (Filter->Frequency * Filter->Frequency2 * M_PI*M_PI * Value);
      if(i & 1) {
        s += QString("L L1 1 %1 660 8 -26 0 1 \"%2H\" 1\n").arg(x).arg(num2str(Value2));
        s += QString("C C1 1 %1 660 -8 46 0 1 \"%2F\" 1\n").arg(x-30).arg(num2str(Value));
        s += QString("GND * 1 %1 690 0 0 0 0\n").arg(x);
      }
      else {
        s += QString("L L1 1 %1 580 -26 -44 0 0 \"%2H\" 1\n").arg(x+40).arg(num2str(Value));
        s += QString("C C1 1 %1 580 -26 10 0 0 \"%2F\" 1\n").arg(x-20).arg(num2str(Value2));
      }
    }
  }


  x += 70;
  if((Filter->Order & 1) == 0)
    x += 70;
  s += QString("Pac lowFreq 1 %1 320 18 -26 0 0 \"2\" 1 \"%2\" 1 \"0\" 0\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 350 0 0 0 0\n").arg(x);
  s += QString("Pac highFreq 1 %1 490 18 -26 0 0 \"3\" 1 \"%2\" 1 \"0\" 0\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 520 0 0 0 0\n").arg(x);
  if(isTriplexer) {
    s += QString("Pac midFreq 1 %1 660 18 -26 0 0 \"4\" 1 \"%2\" 1 \"0\" 0\n").arg(x).arg(Filter->Impedance);
    s += QString("GND * 1 %1 690 0 0 0 0\n").arg(x);
  }

  if(isTriplexer)
    y = 760;
  else
    y = 570;
  Value = Filter->Frequency / 10.0;
  if(isTriplexer)
    Value2 = 10.0 * Filter->Frequency2;
  else
    Value2 = 10.0 * Filter->Frequency;
  s += QString(".SP SP1 1 70 %1 0 67 0 0 \"log\" 1 \"%2Hz\" 1 \"%3Hz\" 1 \"500\" 1 \"no\" 0 \"1\" 0 \"2\" 0\n").arg(y).arg(num2str(Value)).arg(num2str(Value2));
  s += QString("Eqn Eqn1 1 230 %1 0 8 0 0 \"S11_dB=dB(S[1,1])\" 1 \"S21_dB=dB(S[2,1])\" 1 \"S31_dB=dB(S[3,1])\" 1 ").arg(y+17);
  if(isTriplexer)
    s += QString("\"S41_dB=dB(S[4,1])\" 1 ");
  s += QString("\"yes\" 0\n");
  s += "</Components>\n";

  s += "<Wires>\n";

  // connect left source
  x = 60;
  s += QString("60 240 60 290 \"\" 0 0 0\n");
  s += QString("60 240 140 240 \"\" 0 0 0\n");
  s += QString("140 240 140 410 \"\" 0 0 0\n");
  if(isTriplexer) {
    s += QString("120 410 140 410 \"\" 0 0 0\n");
    s += QString("120 410 120 580 \"\" 0 0 0\n");
  }

  // wires down to shunt components
  x = 100;
  for(i = 0; i < (Filter->Order / 2) + 1; i++) {
    x += 140;
    s += QString("%1 240 %2 290 \"\" 0 0 0\n").arg(x).arg(x); // low-pass
    s += QString("%1 410 %2 460 \"\" 0 0 0\n").arg(x).arg(x); // high-pass
    if(isTriplexer)
      s += QString("%1 580 %2 630 \"\" 0 0 0\n").arg(x).arg(x); // band-pass
  }

  // horizontal wires for series components
  for(i = 0; i < (Filter->Order / 2); i++) {
    x = 200 + (i * 140);
    s += QString("%1 240 %2 240 \"\" 0 0 0\n").arg(x).arg(x+80); // low-pass
    s += QString("%1 410 %2 410 \"\" 0 0 0\n").arg(x).arg(x+80); // high-pass
    if(isTriplexer) {
      s += QString("%1 580 %2 580 \"\" 0 0 0\n").arg(x+40).arg(x+60); // series
      s += QString("%1 630 %2 630 \"\" 0 0 0\n").arg(x+10).arg(x+40); // parallel
      s += QString("%1 690 %2 690 \"\" 0 0 0\n").arg(x+10).arg(x+40); // parallel
    }
  }

  // connect right source
  if(Filter->Order & 1) {
    x += 140;
    s += QString("%1 240 %2 240 \"\" 0 0 0\n").arg(x).arg(x+40); // low-pass
    s += QString("%1 410 %2 410 \"\" 0 0 0\n").arg(x).arg(x+40); // high-pass
  }
  else {
    x += 80;
    s += QString("%1 240 %2 240 \"\" 0 0 0\n").arg(x).arg(x+100); // low-pass
    s += QString("%1 410 %2 410 \"\" 0 0 0\n").arg(x).arg(x+100); // high-pass
    if(isTriplexer)
      s += QString("%1 580 %2 580 \"\" 0 0 0\n").arg(x-20).arg(x+100); // band-pass
  }

  s += "</Wires>\n";

  s += "<Diagrams>\n</Diagrams>\n";

  s += "<Paintings>\n";

  s += QString("Text 400 %1 12 #000000 0 \"").arg(y);
  s += Filter::getDescription(Filter);
  s += QString(" \\n impedance matching %1\\ohm\"\n").arg(Filter->Impedance);
  s += "</Paintings>\n";

  return s.toUtf8();
}
