/***************************************************************************
  copyright: (C) 2010 by Michael Margraf
 ***************************************************************************/

#include <QString>
#include <QMessageBox>

#include "config.h"
#include "line_filter.h"

// capacitive end-coupled, half-wavelength bandpass filter
Line_Filter::Line_Filter()
{
}

// -------------------------------------------------------------------------
QByteArray Line_Filter::createSchematic(tFilter *Filter, tSubstrate *Substrate)
{
  double *filterValues, Value, Value2, gap, len, dl, width, er_eff, Wh;
  double Omega = (Filter->Frequency2 + Filter->Frequency) / 2.0;
  double Bandwidth = M_PI_2 * fabs(Filter->Frequency2 - Filter->Frequency) / Omega;

  // get "width" and "er_eff"
  width = er_eff = Wh = 1.0;
  if(Substrate) {
    getMicrostrip(Filter->Impedance, Omega, Substrate, width, er_eff);
    Wh = width / Substrate->height;
  }

  Omega *= M_TWO_PI;  // angular frequency

  filterValues = getNormValues(Filter);
  if(filterValues == 0)
    return QByteArray();  // empty array means errors

  QString s(QUCSFILE_ID PACKAGE_VERSION ">\n<Components>\n");

  int i, x, yc;
  x = 60;
  s += QString("Pac P1 1 %1 320 18 -26 0 0 \"1\" 1 \"%2\" 1\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 350 0 0 0 0\n").arg(x);

  x -= 30;
  yc = 180;
  Value2 = len = dl = 0.0;
  for(i=0; i<=Filter->Order; i++) {
    x += 90;

    // characteristic admittance of J-inverter (normalized to Y0)
    if(i == 0)
      Value = sqrt(Bandwidth / filterValues[1]);
    else if(i == Filter->Order)
      Value = sqrt(Bandwidth / filterValues[i]);
    else
      Value = Bandwidth / sqrt(filterValues[i] * filterValues[i+1]);

    Value /= (1.0 - Value*Value);             // susceptance
    gap = Value / Filter->Impedance / Omega;  // gap capacitance

    if(i > 0) {
      len = LIGHTSPEED / sqrt(er_eff) / Omega
          * (M_PI - 0.5*(atan(2.0*Value) + atan(2.0*Value2)));
      len -= dl;  // last end capacitance (for microstrip only)
    }

    if((Substrate!=0) && (i>0) && (i<Filter->Order)) {
      // calculate gap from gap capacitance
      gap /= 0.04598e-12 * (0.03 + pow(Wh, 1.23)) * (0.272 + 0.07*Substrate->er);
      gap /= (1.0 + 4.19 * (1.0 - exp(-0.785/sqrt(Wh))));
      gap /= 500.0 * Substrate->height;
      gap = log(gap) * Substrate->height / -1.86;

      if(gap < 1e-7) {
        QMessageBox::critical(0, QObject::tr("Error"),
           QObject::tr("The filter can't be created because the bandwidth is too large.\n"
           "You may increase the frequency, the substrate thickness or reduce the permittivity."));
        return QByteArray();
      }

      dl = 0.107 * (Wh+9.0) * pow(gap/Substrate->height, 3.23)
         + 2.09*pow(gap/Substrate->height, 1.05) * (1.5+0.3*Wh) / (1.0+0.6*Wh);
      dl = (dl + 2.04e-5) / (dl + 1.0);
      dl *= getMicrostripOpen(Wh, Substrate->er, er_eff) * Substrate->height;
      len -= dl;
    }

    if(i > 0) {
      if(Substrate)
        s += QString("MLIN MS1 1 %1 %2 -26 15 0 0 \"Subst1\" 0 \"%3\" 1 \"%4\" 1\n").arg(x).arg(yc).arg(num2str(width, "m")).arg(num2str(len, "m"));
      else
        s += QString("TLIN Line1 1 %1 %2 -26 20 0 0 \"%3\" 1 \"%4\" 1\n").arg(x).arg(yc).arg(Filter->Impedance).arg(num2str(len, "m"));
      x += 90;
    }

    if((Substrate!=0) && (i>0) && (i<Filter->Order))
      s += QString("MGAP MS1 1 %1 %2 -26 15 0 0 \"Subst1\" 0 \"%3\" 1 \"%4\" 1 \"%5\" 1\n").arg(x).arg(yc).arg(num2str(width, "m")).arg(num2str(width, "m")).arg(num2str(gap, "m"));
    else
      s += QString("C C1 1 %1 %2 -26 17 0 0 \"%3\" 1\n").arg(x).arg(yc).arg(num2str(gap));

    Value2 = Value;  // remember for next loop
  }


  x += 80;
  s += QString("Pac P2 1 %1 320 18 -26 0 0 \"2\" 1 \"%2\" 1 \"0\" 0\n").arg(x).arg(Filter->Impedance);
  s += QString("GND * 1 %1 350 0 0 0 0\n").arg(x);

  Value  = 2.0 * Filter->Frequency  - Filter->Frequency2;
  Value2 = 2.0 * Filter->Frequency2 - Filter->Frequency;
  s += QString(".SP SP1 1 70 400 0 67 0 0 \"lin\" 1 \"%2Hz\" 1 \"%3Hz\" 1 \"500\" 1 \"no\" 0 \"1\" 0 \"2\" 0\n").arg(num2str(Value)).arg(num2str(Value2));
  if(Substrate)
    s += QString("SUBST Subst1 1 250 440 -30 24 0 0 \"%1\" 1 \"%2\" 1 \"%3\" 1 \"%4\" 1 \"%5\" 1 \"0\" 0\n").arg(Substrate->er).arg(num2str(Substrate->height, "m")).arg(num2str(Substrate->thickness, "m")).arg(Substrate->tand).arg(Substrate->resistivity);
  s += QString("Eqn Eqn1 1 350 477 0 8 0 0 \"S21_dB=dB(S[2,1])\" 1 \"S11_dB=dB(S[1,1])\" 1 \"S21_phase=wphase(S[2,1])\" 1 \"yes\" 0\n");
  s += "</Components>\n";

  s += "<Wires>\n";

  // connect left source
  s += QString("60 180 60 290 \"\" 0 0 0\n");
  s += QString("60 180 90 180 \"\" 0 0 0\n");

  // connect right source
  s += QString("%1 180 %2 290 \"\" 0 0 0\n").arg(x).arg(x);
  s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x-50).arg(x);

  // wires between components
  x = 150;
  for(i = 0; i < 2*Filter->Order; i++) {
    s += QString("%1 180 %2 180 \"\" 0 0 0\n").arg(x).arg(x+30);
    x += 90;
  }

  s += "</Wires>\n";

  s += "<Diagrams>\n</Diagrams>\n";

  s += "<Paintings>\n";
  s += QString("Text 350 400 12 #000000 0 \"");
  s += Filter::getDescription(Filter);
  s += QString(" \\n impedance matching %1 ohms\"\n").arg(Filter->Impedance);
  s += "</Paintings>\n";

  return s.toUtf8();
}
